// ------------------------------------------------------------
// ARMv8 MPCore EL3 AArch64 Startup Code
//
// Basic Vectors, MMU, caches and GICv3 initialization
//
// Exits in EL1 AArch64
//
//
// Copyright (C) 2014-2025 by Arm Limited (or its affiliates). All rights reserved.
// Use, modification and redistribution of this file is subject to your possession of a
// valid End User License Agreement for the Arm Product of which these examples are part of 
// and your compliance with all applicable terms and conditions of such license agreement.
//
// Redistribution permitted only in object code form and only as
// part of Software Applications developed by you or your permitted users. If you
// choose to redistribute the whole or any part of Example Code, you agree: (a)
// to ensure that they are licensed for use only as part of Software Applications
// and only for execution on microprocessors manufactured or simulated under
// licence from Arm; (b) not to use Arm's or any of its licensors names, logos or
// trademarks to market Software Applications; (c) to include valid copyright
// notices on Software Applications, and preserve any copyright notices which are
// included with, or in, the Example Code; and (d) to ensure that any further
// redistribution is limited to redistribution by either or both your customers
// and your authorised distributors as part of Software Applications and that
// your customers and your authorised distributors comply with these terms.
//
// ------------------------------------------------------------

#include "layout.h"
#include "target.h"
#include "v8_mmu.h"
#include "v8_system.h"


    .section StartUp, "ax"
    .balign 4

#ifdef __ARMCC_VERSION
    // use separate stack and heap, as anticipated by layout.scat
    .global __use_two_region_memory
#endif


// ------------------------------------------------------------

    .global start64
    .type start64, "function"
start64:

    //
    // VMPIDR_EL2 holds the value of the Virtualization Multiprocessor ID. This is the value returned by Non-secure EL1 reads of MPIDR_EL1.
    //  VPIDR_EL2 holds the value of the Virtualization Processor ID. This is the value returned by Non-secure EL1 reads of MIDR_EL1.
    // Both of these registers are architecturally UNKNOWN at reset, and so they must be set to the correct value
    // (even if EL2/virtualization is not being used), otherwise non-secure EL1 reads of MPIDR_EL1/MIDR_EL1 will return garbage values.
    // This guarantees that any future reads of MPIDR_EL1 and MIDR_EL1 from Non-secure EL1 will return the correct value.
    //
    // keep MPIDR_EL1.Aff0 (i.e. the CPU no. on Cortex-A cores) in
    // x19 (defined by the AAPCS as callee-saved), so we can re-use
    // the number later
    //
    mrs x0, MPIDR_EL1
    msr VMPIDR_EL2, x0
    bl mpidr_to_core_number
    mov x19, x0

    //
    // Setup some EL3/EL2 stack space, ready for calling some subroutines, below.
    //
    // Stack space allocation is core-specific, so use core
    // number already held in x19
    //
    // 2^12 bytes per core for the EL3 stacks
    //
    ldr x0, =__el3_stack
    sub x0, x0, x19, lsl #12
    mov sp, x0

    // Now skip the EL3 setup if we started at EL2
    mrs x0, CurrentEL
    cmp x0, #(3 << 2)
    b.ne el2_setup

    ldr x1, =el3_vectors
    msr VBAR_EL3, x1

    //
    // set lower exception levels as non-secure, with no access
    // back to EL2 or EL3, and are AArch64 capable
    //
    mov x3, #(SCR_EL3_RW  | \
              SCR_EL3_SMD | \
              SCR_EL3_NS)
    msr SCR_EL3, x3

    msr CPTR_EL3, xzr

    //
    // SCTLR_EL3 comes out of reset with a limited set of known fields
    // set; set the unknowns to 0 (all the other known values are already
    //  0)
    //
#if __BYTE_ORDER__ == __ORDER_BIG_ENDIAN__
    ldr x0, =SCTLR_ELx_EE
#else
    mov x0, xzr
#endif
    msr SCTLR_EL3, x0

    // Program the system counter frequency for the lower exceptions level to read
    ldr x0, =CNT_CONTROL_BASE_ADDRESS
    ldr x1, =0x3        // halt-on-debug and enable
    str w1, [x0]        // CNTControlBase.CNTCR
    ldr x0, [x0, 0x20]  // CNTControlBase.CNTFID0
    msr CNTFRQ_EL0, x0

    //
    // Configure ACTLR_EL3
    // ----------------------
    //
    // These bits are IMPLEMENTATION DEFINED, so are different for
    // different processors
    //
    // For Cortex-A57, the controls we set are:
    //
    //  Enable lower level access to CPUACTLR_EL1
    //  Enable lower level access to CPUECTLR_EL1
    //  Enable lower level access to L2CTLR_EL1
    //  Enable lower level access to L2ECTLR_EL1
    //  Enable lower level access to L2ACTLR_EL1
    //
#define CORTEX_ACTLR ((1 << 0) | \
                      (1 << 1) | \
                      (1 << 4) | \
                      (1 << 5) | \
                      (1 << 6))
#ifdef CORTEXA
    ldr x0, =CORTEX_ACTLR
    msr ACTLR_EL3, x0

    //
    // configure CPUECTLR_EL1
    //
    // These bits are IMP DEF, so need to different for different
    // processors
    //
    // SMPEN - bit 6 - Enables the processor to receive cache
    //                 and TLB maintenance operations
    //
    // Note: For Cortex-A57/53 SMPEN should be set before enabling
    //       the caches and MMU, or performing any cache and TLB
    //       maintenance operations.
    //
    //       This register has a defined reset value, so we use a
    //       read-modify-write sequence to set SMPEN
    //
    mrs x0, S3_1_c15_c2_1  // Read EL1 CPU Extended Control Register
    orr x0, x0, #(1 << 6)  // Set the SMPEN bit
    msr S3_1_c15_c2_1, x0  // Write EL1 CPU Extended Control Register
#endif // CORTEXA

    // Even though we are using the GICv3/4 in legacy GICv2 mode
    // we still need to do GICv3 power management
    ldr x0, =(GICC_BASE + 0xFC)
    ldr w1, [x0]
    ubfx w1, w1, #16, #4 // GIC version

    cmp w1, #3
    b.lt 2f

#ifdef GICR_BASE
    // Wake the GICR
    ldr x0, =(GICR_BASE + 0x14)
    b.gt 4f
    add x0, x0, x19, lsl #17 // In GICv3 each redistributor is two 64kB frames
    b 3f
4:
    add x0, x0, x19, lsl #18 // In GICv4 each redistributor is four 64kB frames
3:
    ldr w1, [x0]
    bic w1, w1, #0x2 // Clear ProcessorSleep
    str w1, [x0]

    // Wait for the children to wake
1:
    ldr w1, [x0]
    and w1, w1, #0x4 // ChildrenAsleep bit
    cbnz w1, 1b
#else
    b . // ERROR: We have a GICv3/4 but GICR_BASE wasn't defined
#endif

2:

    //
    // we need to configure the GIC while still in secure mode, specifically
    // all PPIs and SPIs have to be programmed as Group1 interrupts
    // this is a per-CPU configuration for these interrupts
    //
    mov w0, #0
    mov w1, #1
    bl  SetBlockGroup

    //
    // While we're in the Secure World, set the priority mask low enough
    // for it to be writable in the Non-Secure World
    //
    mov w0, #0xFF
    bl  SetPriorityMask

    //
    // there's more GIC setup to do, but only for the primary CPU
    //
    cmp x19, #PRIMARY_CORE
    b.ne el2_setup

    //
    // There's more to do to the GIC - call the utility routine to set
    // all SPIs to Group1
    //
    mov w0, #1
    bl  SetSPIGroup

el2_setup:

    mrs x0, MIDR_EL1
    msr VPIDR_EL2, x0

    //
    // program the VBARs
    //
    ldr x1, =el1_vectors
    msr VBAR_EL1, x1

    ldr x1, =el2_vectors
    msr VBAR_EL2, x1

    //
    // no traps or VM modifications from the Hypervisor, EL1 is AArch64
    //
    mov x2, #HCR_EL2_RW
    msr HCR_EL2, x2

    // Allow EL1 access to physical counter and timer
    ldr x0, =0x3
    msr CNTHCTL_EL2, x0

    //
    // VMID is still significant, even when virtualisation is not
    // being used, so ensure VTTBR_EL2 is properly initialised
    //
    msr VTTBR_EL2, xzr

    //
    // Don't trap floating point or accesses to CPACR
    //
    msr CPTR_EL2, xzr

    //
    // SCTLR_ELx may come out of reset with UNKNOWN values so we will
    // set the fields to 0 except, possibly, the endianess field(s).
    // Note that setting SCTLR_EL2 or the EL0 related fields of SCTLR_EL1
    // is not strictly needed, since we're never in EL2 or EL0
    //
#ifdef __ARM_BIG_ENDIAN
    mov x0, #(SCTLR_ELx_EE | SCTLR_EL1_E0E)
#else
    mov x0, #0
#endif
    msr SCTLR_EL2, x0
    msr SCTLR_EL1, x0

#ifdef CORTEXA
    ldr x0, =CORTEX_ACTLR
    msr ACTLR_EL2, x0
#endif

    //
    // That's the last of the control settings for now
    //
    // Note: no ISB after all these changes, as registers won't be
    // accessed until after an exception return, which is itself a
    // context synchronisation event
    //


    //
    // Set up EL1 entry point and "dummy" exception return information,
    // then perform exception return to enter EL1
    //
    .global drop_to_el1
drop_to_el1:
    adr x0, el1_entry_aarch64
    mov x1, #(AARCH64_SPSR_EL1h | \
              AARCH64_SPSR_F  | \
              AARCH64_SPSR_I  | \
              AARCH64_SPSR_A)

    mrs x3, CurrentEL
    cmp x3, #(3 << 2)
    b.ne 2f

    msr ELR_EL3, x0
    msr SPSR_EL3, x1
    eret

2:
    msr ELR_EL2, x0
    msr SPSR_EL2, x1
    eret



// ------------------------------------------------------------
// EL1 - Common start-up code
// ------------------------------------------------------------

    .global el1_entry_aarch64
    .type el1_entry_aarch64, "function"
el1_entry_aarch64:

    //
    // Now we're in EL1, setup the application stack
    // the scatter file allocates 2^14 bytes per app stack
    //
    ldr x0, =__stack
    sub x0, x0, x19, lsl #14
    mov sp, x0

    //
    // Enable floating point
    //
    mov x1, #CPACR_EL1_FPEN
    msr CPACR_EL1, x1

    //
    // Invalidate caches and TLBs for all stage 1
    // translations used at EL1
    //
    // Cortex-A processors automatically invalidate their caches on reset
    // (unless suppressed with the DBGL1RSTDISABLE or L2RSTDISABLE pins).
    // It is therefore not necessary for software to invalidate the caches 
    // on startup, however, this is done here in case of a warm reset.
    bl  InvalidateUDCaches
    tlbi VMALLE1


    //
    // Set TTBR0 Base address
    //
    // The CPUs share one set of translation tables that are
    // generated by CPU0 at run-time
    //
    // TTBR1_EL1 is not used in this example
    //
    ldr x1, =__ttb0_l1
    msr TTBR0_EL1, x1


    //
    // Set up memory attributes
    //
    // These equate to:
    //
    // 0 -> 0b01000100 = 0x00000044 = Normal, Inner/Outer Non-Cacheable
    // 1 -> 0b11111111 = 0x0000ff00 = Normal, Inner/Outer WriteBack Read/Write Allocate
    // 2 -> 0b00000100 = 0x00040000 = Device-nGnRE
    //
    mov  x1, #0xff44
    movk x1, #4, LSL #16    // equiv to: movk x1, #0x0000000000040000
    msr MAIR_EL1, x1


    //
    // Set up TCR_EL1
    //
    // We're using only TTBR0 (EPD1 = 1), and the page table entries:
    //  - are using an 8-bit ASID from TTBR0
    //  - have a 4K granularity (TG0 = 0b00)
    //  - are outer-shareable (SH0 = 0b10)
    //  - are using Inner & Outer WBWA Normal memory ([IO]RGN0 = 0b01)
    //  - map
    //      + 32 bits of VA space (T0SZ = 0x20)
    //      + into a 32-bit PA space (IPS = 0b000)
    //
    //     36   32   28   24   20   16   12    8    4    0
    //  -----+----+----+----+----+----+----+----+----+----+
    //       |    |    |OOII|    |    |    |OOII|    |    |
    //    TT |    |    |RRRR|E T |   T|    |RRRR|E T |   T|
    //    BB | I I|TTSS|GGGG|P 1 |   1|TTSS|GGGG|P 0 |   0|
    //    IIA| P P|GGHH|NNNN|DAS |   S|GGHH|NNNN|D S |   S|
    //    10S| S-S|1111|1111|11Z-|---Z|0000|0000|0 Z-|---Z|
    //
    //    000 0000 0000 0000 1000 0000 0010 0101 0010 0000
    //
    //                    0x    8    0    2    5    2    0
    //
    // Note: the ISB is needed to ensure the changes to system
    //       context are before the write of SCTLR_EL1.M to enable
    //       the MMU. It is likely on a "real" implementation that
    //       this setup would work without an ISB, due to the
    //       amount of code that gets executed before enabling the
    //       MMU, but that would not be architecturally correct.
    //
    ldr x1, =0x0000000000802520
    msr TCR_EL1, x1
    isb

    //
    // x19 already contains the CPU number, so branch to secondary
    // code if we're not on the primary cpu
    //
    cmp x19, #PRIMARY_CORE
    b.ne el1_secondary

    //
    // Fall through to primary code
    //


//
// ------------------------------------------------------------
//
// EL1 - primary CPU init code
//
// This code is run on CPU0, while the other CPUs are in the
// holding pen
//

    .global el1_primary
    .type el1_primary, "function"
el1_primary:

    // We need to wake the secondary cores
    bl wake_secondary_cores

    //
    // We're now on the primary processor in the NS world: turn on
    // the banked GIC distributor enable, ready for individual CPU
    // enables later
    //
    bl  EnableGICD

    //
    // Generate TTBR0 L1
    //
    // at 4KB granularity, 32-bit VA space, table lookup starts at
    // L1, with 1GB regions
    //
    // we are going to create entries pointing to L2 tables for a
    // couple of these 1GB regions, the first of which is the
    // RAM on the VE board model - get the table addresses and
    // start by emptying out the L1 page tables (4 entries at L1
    // for a 4K granularity)
    //
    // x21 = address of L1 tables
    //
    ldr x21, =__ttb0_l1
    mov x0, x21
    mov x1, #(4 << 3)
    bl  ZeroBlock

    //
    // time to start mapping the RAM regions - clear out the
    // L2 tables and point to them from the L1 tables
    //
    // x22 = address of L2 tables, needs to be remembered in case
    //       we want to re-use the tables for mapping peripherals
    //
    ldr x22, =__ttb0_l2_ram
    mov x1, #(512 << 3)
    mov x0, x22
    bl  ZeroBlock

    //
    // Get the start address of RAM (the EXEC region) into x4
    // and calculate the offset into the L1 table (1GB per region,
    // max 4GB)
    //
    // x23 = L1 table offset, saved for later comparison against
    //       peripheral offset
    //
    ldr x4, =__code_start
    ubfx x23, x4, #30, #2

    orr x1, x22, #TT_S1_ATTR_PAGE
    str x1, [x21, x23, lsl #3]

    //
    // we've already used the RAM start address in x4 - we now need
    // to get this in terms of an offset into the L2 page tables,
    // where each entry covers 2MB
    //
    ubfx x2, x4, #21, #9

    //
    // TOP_OF_RAM in the scatter file marks the end of the
    // Execute region in RAM: convert the end of this region to an
    // offset too, being careful to round up, then calculate the
    // number of entries to write
    //
    ldr x5, =__top_of_ram
    sub  x3, x5, #1
    ubfx x3, x3, #21, #9
    add  x3, x3, #1
    sub  x3, x3, x2

    //
    // set x1 to the required page table attributes, then orr
    // in the start address (modulo 2MB)
    //
    // L2 tables in our configuration cover 2MB per entry - map
    // memory as Shared, Normal WBWA (MAIR[1]) with a flat
    // VA->PA translation
    //
    bic x4, x4, #((1 << 21) - 1)
    mov x1, #(TT_S1_ATTR_BLOCK | \
             (1 << TT_S1_ATTR_MATTR_LSB) | \
              TT_S1_ATTR_NS | \
              TT_S1_ATTR_AP_RW_PL1 | \
              TT_S1_ATTR_SH_OUTER | \
              TT_S1_ATTR_AF | \
              TT_S1_ATTR_nG)
    orr x1, x1, x4

    //
    // factor the offset into the page table address and then write
    // the entries
    //
    add x0, x22, x2, lsl #3

loop1:
    subs x3, x3, #1
    str x1, [x0], #8
    add x1, x1, #0x200, LSL #12    // equiv to add x1, x1, #(1 << 21)  // 2MB per entry
    bne loop1


    //
    // now mapping the Peripheral regions - clear out the
    // L2 tables and point to them from the L1 tables
    //
    // The assumption here is that all peripherals live within
    // a common 1GB region (i.e. that there's a single set of
    // L2 pages for all the peripherals). We only use a UART
    // and the GIC in this example, so the assumption is sound
    //
    // x24 = address of L2 peripheral tables
    //
    ldr x24, =__ttb0_l2_periph

    //
    // get a peripheral address (e.g. GICD) address into x4 and calculate
    // the offset into the L1 table
    //
    // x25 = L1 table offset
    //
    ldr x4, =GICD_BASE
    ubfx x25, x4, #30, #2

    //
    // here's the tricky bit: it's possible that the peripherals are
    // in the same 1GB region as the RAM, in which case we don't need
    // to prime a separate set of L2 page tables, nor add them to the
    // L1 tables
    //
    // if we're going to re-use the TTB0_L2_RAM tables, get their
    // address into x24, which is used later on to write the PTEs
    //
    cmp x25, x23
    csel x24, x22, x24, EQ
    b.eq nol2setup

    //
    // Peripherals are in a separate 1GB region, and so have their own
    // set of L2 tables - clean out the tables and add them to the L1
    // table
    //
    mov x0, x24
    mov x1, #512 << 3
    bl  ZeroBlock

    orr x1, x24, #TT_S1_ATTR_PAGE
    str x1, [x21, x25, lsl #3]

nol2setup:
    //
    // set x26 to the required page table attributes
    //
    // L2 tables in our configuration cover 2MB per entry - map
    // memory as NS Device-nGnRE (MAIR[2]) with a flat VA->PA
    // translation
    //
    ldr x26, =(TT_S1_ATTR_BLOCK | \
                (2 << TT_S1_ATTR_MATTR_LSB) | \
                TT_S1_ATTR_NS | \
                TT_S1_ATTR_AP_RW_PL1 | \
                TT_S1_ATTR_AF | \
                TT_S1_ATTR_nG)
    //
    // there's only going to be a single 2MB region for GICD (in
    // x4) - get this in terms of an offset into the L2 page tables
    //
    // with larger systems, it is possible that the GIC redistributor
    // registers require extra 2MB pages, in which case extra code
    // would be required here
    //
    ldr x4, =GICD_BASE
    ubfx x2, x4, #21, #9

    //
    // set x1  to the attributes orr
    // the start address (modulo 2MB)
    //
    bic x4, x4, #((1 << 21) - 1)  // start address mod 2MB
    orr x1, x26, x4

    //
    // only a single L2 entry for this, so no loop as we have for RAM, above
    //
    str x1, [x24, x2, lsl #3]

    //
    // there's only going to be a single 2MB region for GICC (in
    // x4) - get this in terms of an offset into the L2 page tables
    //
    // with larger systems, it is possible that the GIC redistributor
    // registers require extra 2MB pages, in which case extra code
    // would be required here
    //
    ldr x4, =GICC_BASE
    ubfx x2, x4, #21, #9

    //
    // set x1  to the attributes orr
    // the start address (modulo 2MB)
    //
    bic x4, x4, #((1 << 21) - 1)  // start address mod 2MB
    orr x1, x26, x4

    //
    // only a single L2 entry again - write it
    //
    str x1, [x24, x2, lsl #3]

    //
    // we have CS3_PERIPHERALS that include the UART controller
    //
    // Again, the code is making assumptions - this time that the CS3_PERIPHERALS
    // region uses the same 1GB portion of the address space as the GICD,
    // and thus shares the same set of L2 page tables
    //
    // Get CS3_PERIPHERALS address into x4 and calculate the offset into the
    // L2 tables
    //
    ldr x4, =CS3_PERIPHERALS_BASE
    ubfx x2, x4, #21, #9

    //
    // set x1  to the attributes orr
    // the start address (modulo 2MB)
    //
    bic x4, x4, #((1 << 21) - 1)  // start address mod 2MB
    orr x1, x26, x4

    //
    // only a single L2 entry again - write it
    //
    str x1, [x24, x2, lsl #3]


#ifdef STM_CONFIGURATION_REGISTERS
    // STM configuration registers
    ldr x4, =STM_CONFIGURATION_REGISTERS
    ubfx x2, x4, #21, #9

    //
    // set x1  to the attributes orr
    // the start address (modulo 2MB)
    //
    bic x4, x4, #((1 << 21) - 1)  // start address mod 2MB
    orr x1, x26, x4

    //
    // only a single L2 entry again - write it
    //
    str x1, [x0, x2, lsl #3]
#endif // STM_CONFIGURATION_REGISTERS

#ifdef STM_EXTENDED_STIMULUS_PORTS
    // STM extended stimulus ports
    ldr x4, =STM_EXTENDED_STIMULUS_PORTS
    ubfx x2, x4, #21, #9

    //
    // set x1  to the attributes orr
    // the start address (modulo 2MB)
    //
    bic x4, x4, #((1 << 21) - 1)  // start address mod 2MB
    orr x1, x26, x4

    //
    // factor the offset into the page table address and then write
    // the entries
    //
    add x0, x0, x2, lsl #3

    // There are up to 16MB of stimulus ports which is 8 2MB entries
    mov x3, #8
1:
    subs x3, x3, #1
    str x1, [x0], #8
    add x1, x1, #0x200, LSL #12    // equiv to add x1, x1, #(1 << 21)  // 2MB per entry
    bne 1b
#endif // STM_EXTENDED_STIMULUS_PORTS

    //
    // issue a barrier to ensure all table entry writes are complete
    //
    dsb ish

    //
    // Enable the MMU.  Caches will be enabled later, after scatterloading.
    //
    mrs x1, SCTLR_EL1
    orr x1, x1, #SCTLR_ELx_M
    bic x1, x1, #SCTLR_ELx_A // Disable alignment fault checking.  To enable, change bic to orr
    msr SCTLR_EL1, x1
    isb

    b   __main

// ------------------------------------------------------------
// Default implemenation of secondary core wake up
//
// Assumes they don't need to be woken
// ------------------------------------------------------------
   .global wake_secondary_cores
   .type wake_secondary_cores, "function"
   .weak wake_secondary_cores
   ret // Do nothing


// ------------------------------------------------------------
// EL1 - secondary CPU init code
//
// This code is run on CPUs 1, 2, 3 etc....
// ------------------------------------------------------------

    .global el1_secondary
    .type el1_secondary, "function"
el1_secondary:

    //
    // the primary CPU is going to use SGI 15 as a wakeup event
    // to let us know when it is OK to proceed, so prepare for
    // receiving that interrupt
    //
    // NS interrupt priorities run from 0 to 15, with 15 being
    // too low a priority to ever raise an interrupt, so let's
    // use 14
    //
    mov w0, #15
    mov w1, #14 << 4    // we're in NS world, so 4 bits of priority,
                        // 8-bit field, - 4 = 4-bit shift
    bl SetIRQPriority

    mov w0, #15
    bl  EnableIRQ

    //
    // set priority mask as low as possible; although,being in the
    // NS World, we can't set bit[7] of the priority, we still
    // write all 8-bits of priority to an ICC register
    //
    mov x0, #0xFF
    bl SetPriorityMask


    //
    // set global enable and wait for our interrupt to arrive
    //
    bl EnableGICC

loop_wfi:
    dsb SY      // Clear all pending data accesses
    wfi         // Go to sleep

    //
    // something woke us from our wait, was it the required interrupt?
    //
    mov w0, #15
    bl  TestIRQ
    cbz w0, loop_wfi

    //
    // it was - there's no need to actually take the interrupt,
    // so just clear it
    //
    mov w0, #15
    bl  ClearIRQ

    //
    // Enable the MMU and caches
    //
    mrs x1, SCTLR_EL1
    orr x1, x1, #SCTLR_ELx_M
    orr x1, x1, #SCTLR_ELx_C
    orr x1, x1, #SCTLR_ELx_I
    bic x1, x1, #SCTLR_ELx_A // Disable alignment fault checking.  To enable, change bic to orr
    msr SCTLR_EL1, x1
    isb

    //
    // Branch to thread start
    //
    B  MainApp

