/* Copyright (C) 2018-2025 by Arm Limited. All rights reserved. */

#ifndef INCLUDE_LINUX_PROC_PROCESS_CHILDREN_H
#define INCLUDE_LINUX_PROC_PROCESS_CHILDREN_H

#include <cstdint>
#include <map>
#include <optional>
#include <set>
#include <unordered_map>
#include <utility>

#include <unistd.h>

namespace lnx {

    enum class tid_enumeration_mode_t : std::uint8_t {
        /** Only collect the pids specified by the user */
        self_only = 0,
        /** Only collect the pids specified by the user and any of their threads */
        self_and_threads = 1,
        /** Only collect the pids specified by the user and any of their threads, and any other processes they spawn */
        self_and_threads_and_children = 2,
    };

    /**
     * Inherently racey function to collect child tids because threads can be created and destroyed while this is running
     */
    void addTidsRecursively(std::set<int> & tids, int tid, tid_enumeration_mode_t tid_enumeration_mode);

    /**
     * Inherently racey function to collect child tids because threads can be created and destroyed while this is running
     *
     * @return as many of the known child tids (including child processes)
     */
    inline std::set<int> getChildTids(int tid, tid_enumeration_mode_t tid_enumeration_mode)
    {
        std::set<int> result;
        addTidsRecursively(result, tid, tid_enumeration_mode);
        return result;
    }

    enum class sigstop_result_t {
        success,
        failed_no_such_pid,
        failed_other,
    };

    /**
     * send SIGSTOP to some pid
     *
     * @param pid  The pid to signal
     * @return A result indicating what happened
     */
    [[nodiscard]] sigstop_result_t send_sigstop(pid_t pid);

    /** RAII object that sends SIGCONT to some pid on request or dtor */
    class sig_continuer_t {
    public:
        constexpr sig_continuer_t() = default;
        explicit constexpr sig_continuer_t(pid_t pid) : pid(pid) {}

        // not copyable
        sig_continuer_t(sig_continuer_t const &) = delete;
        sig_continuer_t & operator=(sig_continuer_t const &) = delete;

        // only movable
        sig_continuer_t(sig_continuer_t && that) noexcept : pid(std::exchange(that.pid, 0)) {}
        sig_continuer_t & operator=(sig_continuer_t && that) noexcept
        {
            if (this != &that) {
                sig_continuer_t tmp {std::move(that)};
                std::swap(pid, tmp.pid);
            }
            return *this;
        }

        // destructor sends sigcont
        ~sig_continuer_t() noexcept { signal(); }

        /** The stopped pid will not be resumed (probably because this pid was released into another continuer) */
        void discard() { pid = 0; }

        /** sent sigcont to the target pid */
        void signal() noexcept;

    private:
        pid_t pid {0};
    };

    /** The map of found pid to optional paused continuer */
    using pid_to_continuer_t = std::unordered_map<pid_t, std::optional<lnx::sig_continuer_t>>;

    /** Find all the tids associated with a set of pids and sigstop them (so long as the pid is not in the filter set) */
    [[nodiscard]] std::set<pid_t> stop_all_tids(std::set<pid_t> const & pids,
                                                std::set<pid_t> const & filter_set,
                                                std::map<pid_t, sig_continuer_t> & paused_tids,
                                                tid_enumeration_mode_t tid_enumeration_mode);
}

#endif
