/* Copyright (C) 2010-2025 by Arm Limited. All rights reserved. */

#include "SessionData.h"

#include "Configuration.h"
#include "GatorCLIFlags.h"
#include "Logging.h"
#include "SessionXML.h"
#include "lib/SharedMemory.h"

#include <cstring>

#include <unistd.h>

const char MALI_GRAPHICS[] = "\0mali_thirdparty_server";
// Size of the MALI_GRAPHICS socket path, including the null-terminator
const size_t MALI_GRAPHICS_SIZE = sizeof(MALI_GRAPHICS);

SessionData gSessionData;

void SessionData::initialize()
{
    mSharedData = shared_memory::make_unique<SharedData>();
    mWaitingOnCommand = false;
    mLocalCapture = false;
    mOneShot = false;
    mAllowCommands = false;
    mFtraceRaw = false;
    mCaptureOperationMode = CaptureOperationMode::application_default;
    mExcludeKernelEvents = false;
    mEnableOffCpuSampling = false;
    mUseGPUTimeline = GPUTimelineEnablement::automatic;
    mImages.clear();
    mConfigurationXMLPath = nullptr;
    mSessionXMLPath = nullptr;
    mEventsXMLPath = nullptr;
    mEventsXMLAppend = nullptr;
    mTargetPath = nullptr;
    mAPCDir = nullptr;
    mCaptureWorkingDir = nullptr;
    mCaptureUser = nullptr;
    mSampleRate = none;
    mSampleRateGpu = none;
    mSampleRateMetrics = none;
    mMetricsPeriod = -1;
    mLiveRate = 0;
    mDuration = 0;
    mBacktraceDepth = 0;
    mTotalBufferSize = 0;
    long l = sysconf(_SC_PAGE_SIZE);
    if (l < 0) {
        LOG_ERROR("Unable to obtain the page size");
        handleException();
    }
    mPageSize = static_cast<int>(l);
    mAnnotateStart = -1;
    parameterSetFlag = 0;
}

void SessionData::parseSessionXML(char * xmlString)
{
    SessionXML session(xmlString);
    session.parse();

    // Set session data values - use prime numbers just below the desired value to reduce the chance of events firing at the same time
    if ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_SAMPLE_RATE) == 0) {
        if (session.parameters.sample_rate == "high") {
            mSampleRate = high;
            mSampleRateGpu = high;
            mSampleRateMetrics = high_metrics;
        }
        else if (session.parameters.sample_rate == "normal") {
            mSampleRate = normal;
            mSampleRateGpu = normal_gpu; // sample rate will be doubled in normal mode for gpuid >= Valhall
            mSampleRateMetrics = normal_metrics;
        }
        else if (session.parameters.sample_rate == "low") {
            mSampleRate = low;
            mSampleRateGpu = low;
            mSampleRateMetrics = low_metrics;
        }
        else if (session.parameters.sample_rate == "none") {
            mSampleRate = none;
            mSampleRateGpu = none;
            mSampleRateMetrics = none;
        }
        else {
            LOG_ERROR("Invalid sample rate (%s) in session xml.", session.parameters.sample_rate.c_str());
            handleException();
        }
    }

    // set the metrics base period from the CLI flags
    if ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_METRICS_PERIOD) != 0) {
        mMetricsPeriod = gSessionData.mMetricsPeriod;
    }

    if ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_CALL_STACK_UNWINDING) == 0) {
        // NOLINTNEXTLINE(readability-magic-numbers)
        mBacktraceDepth = session.parameters.call_stack_unwinding ? 128 : 0;
    }

    // Determine buffer size (in MB) based on buffer mode
    mOneShot = true;
    if (session.parameters.buffer_mode == "streaming") {
        mOneShot = false;
        mTotalBufferSize = 1; //NOLINT(readability-magic-numbers)
    }
    else if (session.parameters.buffer_mode == "small") {
        mTotalBufferSize = 16; //NOLINT(readability-magic-numbers)
    }
    else if (session.parameters.buffer_mode == "normal") {
        mTotalBufferSize = 64; //NOLINT(readability-magic-numbers)
    }
    else if (session.parameters.buffer_mode == "large") {
        mTotalBufferSize = 256; //NOLINT(readability-magic-numbers)
    }
    else {
        LOG_ERROR("Invalid value for buffer mode in session xml.");
        handleException();
    }

    if ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_GPU_TIMELINE) == 0) {
        if (session.parameters.gpu_timeline == "auto") {
            mUseGPUTimeline = GPUTimelineEnablement::automatic;
        }
        else if (session.parameters.gpu_timeline == "yes") {
            mUseGPUTimeline = GPUTimelineEnablement::enable;
        }
        else if (session.parameters.gpu_timeline == "no") {
            mUseGPUTimeline = GPUTimelineEnablement::disable;
        }
        else {
            LOG_ERROR("Invalid value (%s) for gpu_timeline.", session.parameters.gpu_timeline.c_str());
            handleException();
        }
    }

    if ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_METRIC_SAMPLING_MODE) == 0) {
        if (session.parameters.metric_sampling_mode == "strobing") {
            mMetricSamplingMode = MetricSamplingMode::strobing;
        }
        else if (session.parameters.metric_sampling_mode == "ebs") {
            mMetricSamplingMode = MetricSamplingMode::ebs;
        }
        else if (session.parameters.metric_sampling_mode == "auto") {
            mMetricSamplingMode = MetricSamplingMode::automatic;
        }
        else {
            LOG_ERROR("Invalid value (%s) for metric_sampling_mode.", session.parameters.metric_sampling_mode.c_str());
            handleException();
        }
    }

    mLiveRate = 0;
    if (session.parameters.live_rate > 0) {
        if (mLocalCapture) {
            LOG_DEBUG("Local capture is not compatable with live, disabling live");
        }
        else {
            // Convert milli- to nanoseconds
            mLiveRate = session.parameters.live_rate * 1000000ULL;
        }
    }
    if ((!isCaptureOperationModeSystemWide(mCaptureOperationMode)) && (mWaitForProcessCommand == nullptr)
        && mCaptureCommand.empty() && mPids.empty()) {
        LOG_ERROR("No command specified in Capture and Analysis Options.");
        handleException();
    }

    if ((!mAllowCommands) && (!mCaptureCommand.empty())
        && ((gSessionData.parameterSetFlag & USE_CMDLINE_ARG_CAPTURE_COMMAND) == 0)) {
        LOG_ERROR(
            "Running a command during a capture is not currently allowed. Please restart gatord with the -a flag.");
        handleException();
    }
}
