/* Copyright (C) 2014-2025 by Arm Limited. All rights reserved. */

#ifndef PARSERRESULT_H_
#define PARSERRESULT_H_

#include "Configuration.h"
#include "linux/smmu_identifier.h"
#include "metrics/definitions.hpp"
#include "metrics/metric_group_set.hpp"

#include <map>
#include <optional>
#include <set>
#include <string>
#include <vector>
/**
 * For containing the results of parsing
 */
#define DEFAULT_PORT 8080

class ParserResult {
public:
    enum class ExecutionMode { LOCAL_CAPTURE, PRINT, DAEMON, EXIT, USAGE };

    enum class Printable {
        EVENTS_XML,
        COUNTERS_XML,
        DEFAULT_CONFIGURATION_XML,
        COUNTERS,
        COUNTERS_DETAILED,
        WORKFLOW,
    };

    std::vector<SpeConfiguration> mSpeConfigs;
    std::vector<std::string> mCaptureCommand;
    std::set<int> mPids;
    std::map<std::string, EventCode> events;
    std::set<Printable> printables;
    std::vector<std::string> error_messages;

    // Each entry in this set dictates some metrics to record
    metrics::metric_group_set_t enabled_metric_groups;

    std::uint64_t parameterSetFlag {0};

    ExecutionMode mode {ExecutionMode::DAEMON};

    const char * mCaptureWorkingDir {nullptr};
    const char * mSessionXMLPath {nullptr};
    const char * mTargetPath {nullptr};
    const char * mConfigurationXMLPath {nullptr};
    const char * mEventsXMLPath {nullptr};
    const char * mEventsXMLAppend {nullptr};
    const char * mWaitForCommand {nullptr};
    const char * pmuPath {nullptr};
    const char * mAndroidPackage {nullptr};
    const char * mAndroidActivity {nullptr};
    const char * mAndroidActivityFlags {nullptr};

    gator::smmuv3::default_identifiers_t smmu_identifiers;

    int mBacktraceDepth {0};
    SampleRate mSampleRate {none};
    SampleRate mSampleRateGpu {none};
    SampleRate mSampleRateMetrics {none};
    int mMetricsPeriod {-1};
    int mDuration {0};
    int mPerfMmapSizeInPages {-1};
    int mSpeSampleRate {-1};
    int mOverrideNoPmuSlots {-1};
    int port {DEFAULT_PORT};
    GPUTimelineEnablement mGPUTimelineEnablement {GPUTimelineEnablement::automatic};

    CaptureOperationMode mCaptureOperationMode = CaptureOperationMode::system_wide;
    MetricSamplingMode mMetricMode = MetricSamplingMode::automatic;

    bool mFtraceRaw {false};
    bool mStopGator {false};
    bool mAllowCommands {false};
    bool mDisableCpuOnlining {false};
    bool mDisableKernelAnnotations {false};
    bool mExcludeKernelEvents {false};
    bool mEnableOffCpuSampling {false};
    bool mLogToFile {false};
    bool mHasProbeReportFlag {false};

    /**
     * @return - a list of argument-value pairs
     */
    [[nodiscard]] const std::vector<std::pair<std::string, std::optional<std::string>>> & getArgValuePairs() const;

    /**
     * Add a new argument value pair to the list of argument-value pairs
     */
    void addArgValuePair(std::pair<std::string, std::optional<std::string>> argValuePair);

    /**
     * Move the --app or -A argument to the end of the list of argument value pairs
     */
    void moveAppArgToEndOfVector();

    /**
     * Clears the  list of argument-value pairs and set the ExecutionMode to Exit
     */
    void parsingFailed();

    /**
     * @brief Returns whether the argument parsing has succeeded or not.
     *
     * @return true When the parsing is regarded as successful.
     * @return false When the parsing has failed.  ExecutionMode is EXIT.
     */
    [[nodiscard]] bool ok() const;

    ParserResult() = default;
    ParserResult(const ParserResult &) = delete;
    ParserResult & operator=(const ParserResult &) = delete;
    ParserResult(ParserResult &&) = delete;
    ParserResult & operator=(ParserResult &&) = delete;

private:
    std::vector<std::pair<std::string, std::optional<std::string>>> argValuePairs;
};

#endif /* PARSERRESULT_H_ */
