/*
 *  Streamline Annotation Example
 *
 * Copyright (C) 2011-2025 by Arm Limited (or its affiliates). All rights reserved.
 * Use, modification and redistribution of this file is subject to your possession of a
 * valid End User License Agreement for the Arm Product of which these examples are part of
 * and your compliance with all applicable terms and conditions of such license agreement.
 *
 * Redistribution permitted only in object code form and only as
 * part of Software Applications developed by you or your permitted users. If you
 * choose to redistribute the whole or any part of Example Code, you agree: (a)
 * to ensure that they are licensed for use only as part of Software Applications
 * and only for execution on microprocessors manufactured or simulated under
 * licence from Arm; (b) not to use Arm's or any of its licensors names, logos or
 * trademarks to market Software Applications; (c) to include valid copyright
 * notices on Software Applications, and preserve any copyright notices which are
 * included with, or in, the Example Code; and (d) to ensure that any further
 * redistribution is limited to redistribution by either or both your customers
 * and your authorised distributors as part of Software Applications and that
 * your customers and your authorised distributors comply with these terms.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include "streamline_annotate.h"


char* readFromDisk(const char* file, unsigned int *size) {
    // Open the file
    FILE* pFile;

    ANNOTATE_CHANNEL_COLOR(2, ANNOTATE_WHITE, "");

    pFile = fopen(file, "rb");
    if (pFile == NULL) {
        return NULL;
    }

    // Obtain file size
    fseek(pFile , 0 , SEEK_END);
    unsigned int lSize = ftell(pFile);
    rewind(pFile);

    // Allocate memory to contain the whole file
    char* buffer = (char*)malloc(lSize);
    if (buffer == NULL) return NULL;

    // Copy the file into the buffer
    if (fread(buffer,1,lSize,pFile) != lSize) return NULL;

    // Terminate
    fclose(pFile);

    if (size)
        *size = lSize;

    ANNOTATE_CHANNEL_END(2);

    return buffer;
}

struct bmp_magic {
  unsigned char ch1;
  unsigned char ch2;
};

struct bmp_header {
  uint32_t filesz;
  uint16_t creator1;
  uint16_t creator2;
  uint32_t bmp_offset;
};

struct dib_header {
  uint32_t header_sz;
  int32_t width;
  int32_t height;
  uint16_t nplanes;
  uint16_t bitspp;
  uint32_t compress_type;
  uint32_t bmp_bytesz;
  int32_t hres;
  int32_t vres;
  uint32_t ncolors;
  uint32_t nimpcolors;
};

#define BMP_HDR_SIZE    (sizeof(struct bmp_magic) + sizeof(struct bmp_header) + sizeof(struct dib_header))
#define BMP_WIDTH       100
#define BMP_HEIGHT      100
#define BMP_PAD_BYTES   (BMP_WIDTH * 3 % 4)
#define BMP_ROW_SIZE    (BMP_WIDTH * 3 + BMP_PAD_BYTES)
#define BMP_DATA_SIZE   (BMP_ROW_SIZE * BMP_HEIGHT)
#define BMP_FILE_SIZE   (BMP_HDR_SIZE + BMP_DATA_SIZE)

int addBmpHeader(char* buffer) {
    struct bmp_magic bmpmagic;
    struct bmp_header bmpheader;
    struct dib_header dibheader;

    ANNOTATE_CHANNEL_COLOR(3, ANNOTATE_BLACK, "");

    // BMP magic
    bmpmagic.ch1 = 0x42;
    bmpmagic.ch2 = 0x4D;

    // BMP header
    bmpheader.filesz = BMP_FILE_SIZE;
    bmpheader.creator1 = 0;
    bmpheader.creator2 = 0;
    bmpheader.bmp_offset = BMP_HDR_SIZE;

    // DIB header
    dibheader.header_sz = sizeof(dibheader);
    dibheader.width = BMP_WIDTH;
    dibheader.height = BMP_HEIGHT;
    dibheader.nplanes = 1;
    dibheader.bitspp = 24;
    dibheader.compress_type = 0;
    dibheader.bmp_bytesz = BMP_DATA_SIZE;
    dibheader.hres = 2835;
    dibheader.vres = 2835;
    dibheader.ncolors = 0;
    dibheader.nimpcolors = 0;

    // Create the bmp image header by copying the structures into a single buffer
    memcpy(buffer, (char*)&bmpmagic, sizeof(bmpmagic));
    memcpy(&buffer[sizeof(bmpmagic)], (char*)&bmpheader, sizeof(bmpheader));
    memcpy(&buffer[sizeof(bmpmagic) + sizeof(bmpheader)], (char*)&dibheader, sizeof(dibheader));

    ANNOTATE_CHANNEL_END(3);

    return BMP_HDR_SIZE;
}

void render(char* buffer, int loop) {
    int offset, row, col, red, green, blue, sq_size;
    char square[3] = {0x60, 0x10, 0x10};
    char pad[3] = {0, 0, 0};
    static int dataOffset;

    ANNOTATE_CHANNEL_COLOR(4, ANNOTATE_BLUE, "");

    // Create the bmp header once and reuse the same buffer
    if (loop == 0)
        dataOffset = addBmpHeader(buffer);
    offset = dataOffset;

    // Shade the background
    red = 0x80;
    green = 0xe0 - loop;
    blue = 0x20 + loop;
    for (row = 0; row < BMP_HEIGHT; row++) {
        for (col = 0; col < BMP_WIDTH; col++) {
            buffer[offset++] = blue & 0xff;
            buffer[offset++] = green & 0xff;
            buffer[offset++] = red & 0xff;
        }
        memcpy(&buffer[offset], pad, BMP_PAD_BYTES);
        offset += BMP_PAD_BYTES;
        blue++;
        green--;
    }

    // Reset the offset
    offset = dataOffset;

    // Choose the lesser of width or height
    sq_size = BMP_HEIGHT > BMP_WIDTH ? BMP_WIDTH : BMP_HEIGHT;

    // Bound the size of the square by sq_size
    sq_size = loop % sq_size;

    // Draw a square over the background
    for (row = 0; row < sq_size; row++) {
        for (col = 0; col < sq_size; col++) {
            memcpy(&buffer[offset + col * 3], square, 3);
        }
        // Change the red value of the square
        square[2] = (((int)square[2]) + 1) & 0xff;
        offset += BMP_ROW_SIZE;
    }

    ANNOTATE_CHANNEL_END(4);
}

void displayImage() {
    char filename[32];
    char* image;
    unsigned int size;

    ANNOTATE_CHANNEL_COLOR(1, ANNOTATE_GREEN, "");

    // Supported formats include gif, png, jpeg, tiff, ico, bmp (+rle)
    strcpy(filename, "splash.bmp");
    image = readFromDisk(filename, &size);
    if (image == NULL) {
        printf("error loading image %s\n", filename);
        exit(1);
    }

    // Add text along with the image annotation
    ANNOTATE_VISUAL(image, size, filename);
    free(image);

    ANNOTATE_CHANNEL_END(1);
}

int main(int argc, char** argv) {
    char renderstr[32];
    char buffer[BMP_FILE_SIZE];
    int loop, max = 200;

    if (argc > 1 && argv)
        max = atoi(argv[1]);

    ANNOTATE_SETUP;
    ANNOTATE_NAME_GROUP(1, "debug");
    ANNOTATE_NAME_CHANNEL(1, 1, "displayImage");
    ANNOTATE_NAME_CHANNEL(2, 1, "readFromDisk");
    ANNOTATE_NAME_CHANNEL(3, 1, "addBmpHeader");
    ANNOTATE_NAME_CHANNEL(4, 1, "render");

    // annotate the splash image
    displayImage();

    ANNOTATE_MARKER_COLOR_STR(ANNOTATE_CYAN, "End of splash image, start rendering bmp images");

    for (loop = 0; loop < max; loop++) {
        snprintf(renderstr, sizeof(renderstr), "render %d", loop);
        ANNOTATE_COLOR(ANNOTATE_PURPLE, renderstr);
        render(buffer, loop);
        ANNOTATE_VISUAL(buffer, BMP_FILE_SIZE, "");
    }

    return 0;
}
