/**
 * Copyright (C) 2006-2025 by Arm Limited (or its affiliates). All rights reserved.
 * Use, modification and redistribution of this file is subject to your possession of a
 * valid End User License Agreement for the Arm Product of which these examples are part of
 * and your compliance with all applicable terms and conditions of such license agreement.
 *
 * Redistribution permitted only in object code form and only as
 * part of Software Applications developed by you or your permitted users. If you
 * choose to redistribute the whole or any part of Example Code, you agree: (a)
 * to ensure that they are licensed for use only as part of Software Applications
 * and only for execution on microprocessors manufactured or simulated under
 * licence from Arm; (b) not to use Arm's or any of its licensors names, logos or
 * trademarks to market Software Applications; (c) to include valid copyright
 * notices on Software Applications, and preserve any copyright notices which are
 * included with, or in, the Example Code; and (d) to ensure that any further
 * redistribution is limited to redistribution by either or both your customers
 * and your authorised distributors as part of Software Applications and that
 * your customers and your authorised distributors comply with these terms.
 */
/*
 * This file contains the main() program that sets the vector table location, displays a welcome message,
 * initializes the MPU, starts the SysTick timer, initializes the Process Stack Pointer, changes Thread mode
 * to Unprivileged and to use the Process Stack, then runs the main application
 */

#include "intrinsics.h"
#include "barman.h"
#include "barman-example.h"
#include "device.h"

#include <stdio.h>
#include <stdarg.h>

/*
 * Initialize barman and start sampling.
 *
 * Must be called in privileged mode.
 */
void initBarman(void)
{
    const struct bm_protocol_clock_info clock_info = {
        .timestamp_base = 0,
        .timestamp_multiplier = 1000000000,
        .timestamp_divisor = PROCESSOR_CLOCK_FREQ_HZ,
        .unix_base_ns = 0
    };

#if BM_CONFIG_MAX_TASK_INFOS > 0
#define NUM_TASKS 5
    const struct bm_protocol_task_info task_entries[NUM_TASKS] =
    {
        {TASK_ID_INIT, "init"},
        {TASK_ID_INSERTION_SORT, "insertion sort"},
        {TASK_ID_SHELL_SORT, "shell sort"},
        {TASK_ID_QUICK_SORT, "quick sort"},
        {TASK_ID_INTERRUPT, "interrupt"}
    };
#endif

    /* Initialize barman but if there is a problem we will loop here */
    while (!barman_initialize_with_itm_interface("barman M profile example on " STRINGIFY_VAL(TARGET), &clock_info,
#if BM_CONFIG_MAX_TASK_INFOS > 0
                                                 NUM_TASKS, task_entries,
#endif
#if BM_CONFIG_MAX_MMAP_LAYOUTS > 0
                                                /* We only have one image for all tasks so we don't need to provide these */
                                                0, BM_NULL,
#endif
                              1));

    /* Now we are ready to enable sampling */
    barman_enable_sampling();
}

/*
 * If we want to provide Streamline with task information
 * we must provide barman with a way to get the current task ID
 */
static bm_task_id_t task_id = TASK_ID_INIT;

bm_task_id_t barman_ext_get_current_task_id(void)
{
    /* In this example we have one task running on each core */
    return task_id;
}

void switchTask(enum task_id new_task_id)
{
    task_id = new_task_id;
#if BM_CONFIG_MAX_TASK_INFOS > 0
    barman_record_task_switch(BM_TASK_SWITCH_REASON_PREEMPTED);
#endif
}

enum task_id getTaskId()
{
    return task_id;
}

/*
 * If we decide to enable logging within barman,
 * just print the messages using semihosting.
 */
#if BM_CONFIG_ENABLE_DEBUG_LOGGING
void barman_ext_log_debug(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}
#endif

#if BM_CONFIG_ENABLE_LOGGING
void barman_ext_log_error(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}

void barman_ext_log_warning(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}

void barman_ext_log_info(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}
#endif
