/** * MEERKAT MAIN PROCESSING SCRIPT ** */

/* Global data load */
var jsonData = JSON.parse($("#meerkatData").html());
//console.log(jsonData);
// Used for generating unique div IDs
var counter = 0;
var headingCounter = 0;
var textCounter = 0;
var adviceCounter = 0;

var interactiveChart;
var decimalPlaces = 0;

// Cache for xvalue arrays as they are common for all graphs within a region
const xdataCache = [];

const INFO_IMAGE = "data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4gIDxnPiAgPHRpdGxlPmJhY2tncm91bmQ8L3RpdGxlPiAgPHJlY3QgZmlsbD0ibm9uZSIgaWQ9ImNhbnZhc19iYWNrZ3JvdW5kIiBoZWlnaHQ9IjEwMiIgd2lkdGg9IjEwMiIgeT0iLTEiIHg9Ii0xIi8+ICA8ZyBkaXNwbGF5PSJub25lIiBvdmVyZmxvdz0idmlzaWJsZSIgeT0iMCIgeD0iMCIgaGVpZ2h0PSIxMDAlIiB3aWR0aD0iMTAwJSIgaWQ9ImNhbnZhc0dyaWQiPiAgIDxyZWN0IGZpbGw9InVybCgjZ3JpZHBhdHRlcm4pIiBzdHJva2Utd2lkdGg9IjAiIHk9IjAiIHg9IjAiIGhlaWdodD0iMTAwJSIgd2lkdGg9IjEwMCUiLz4gIDwvZz4gPC9nPiA8Zz4gIDx0aXRsZT5MYXllciAxPC90aXRsZT4gIDxwYXRoIGlkPSJzdmdfMiIgZD0ibTQ5LjYzMjIyMSwxLjA4NTA1OWMtMjcuMDYxODg0LDAgLTQ4Ljk5OTk1MiwyMS45MzgwNzMgLTQ4Ljk5OTk1Miw0OC45OTk5NjNjMCwyNy4wNjI0MjQgMjEuOTM4MDY4LDQ4Ljk5OTk2MiA0OC45OTk5NTIsNDguOTk5OTYyYzI3LjA2MjQ4MSwwIDQ5LjAwMDA0MiwtMjEuOTM3NTM4IDQ5LjAwMDA0MiwtNDguOTk5OTYyYzAsLTI3LjA2MTg5IC0yMS45Mzc1NjEsLTQ4Ljk5OTk2MyAtNDkuMDAwMDQyLC00OC45OTk5NjN6bS00LjE5MDcyLDIyLjAyODc3Mmw4LjE1ODk1OCwwbDAsOC42Nzg1MjRsLTguMTU4OTU4LDBsMCwtOC42Nzg1MjR6bTEzLjUwMjY3OCw1Mi42Njg0ODhsLTguOTc0MzI3LDBjLTMuNDg2NjE4LDAgLTQuOTczMzU4LC0xLjQ4MzQ5OCAtNC45NzMzNTgsLTUuMDQ0MDIybDAsLTIzLjE0NDQ2M2MwLC0xLjExMjQ1MyAtMC41OTI4OTksLTEuNjMyMDQyIC0xLjYzMTQyOCwtMS42MzIwNDJsLTIuOTY2OTYxLDBsMCwtOC4wMTM2MDdsOC45NzQ5NDUsMGMzLjQ4OTIzMSwwIDQuOTY5NDk0LDEuNTU3NDA0IDQuOTY5NDk0LDUuMDQzMzg4bDAsMjMuMjE5MDU5YzAsMS4wNDE3NCAwLjU5Mjg4LDEuNjMyMDUgMS42MzE0MjQsMS42MzIwNWwyLjk2Njk4LDBsMCw3LjkzOTYzNmwwLjAwMzIzMSwweiIgc3Ryb2tlLXdpZHRoPSIwIiBzdHJva2U9IiMwMDAiIGZpbGw9IiMwMDkxYmQiLz4gPC9nPjwvc3ZnPg=="
const SPANNER_IMG = "data:image/svg+xml;base64,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"
const BACKTOTOP_IMG = "data:image/svg+xml;base64,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";

var htmlContent;
// Add the html output in the content section of the page
$("#content").html(htmlContent);

var chartGroups = {overall: []};

parseJson();
jsonData = null;

var binToFrameMapper;

function showFullResolutionImage(imageSrc) {
  var overlay = document.getElementById('overlay');
  var overlayImage = document.getElementById('overlay-image');
  var overlayClose = document.getElementById('overlay-close');

  overlayImage.src = imageSrc;
  overlay.style.display = 'block';

  overlay.addEventListener('click', function (e) {
    if (e.target === overlay) {
      hideOverlay();
    }
  });
  overlayClose.addEventListener('click', function (e) {
    hideOverlay();
  });
}

function hideOverlay() {
  var overlay = document.getElementById('overlay');
  overlay.style.display = 'none';
}

function parseJson() {
  $.each(jsonData, function(name, element) {
    if(element.excludeFromHtml) {
      return true; // continue
    }
    switch(element.type) {
      case 'FRAME_MAPPER':
        binToFrameMapper = element;
        break;
      case 'TEXT':
      case 'KEY_VALUE':
        insertText(element);
        break;
      case 'DUAL_AXIS':
        insertDualAxis(element);
        break;
      case 'GAUGE':
        insertGauge(element);
        break;
      case 'BAR':
        insertBar(element);
        break;
      case 'INTERACTIVE':
        // Only allow one interactive chart
        if (!interactiveChart) {
          insertInteractive(element);
        } else {
          console.log("Only one interactive chart per report")
        }
        break;
      case 'GROUP':
        parseGroup(element.elements);
        break;
      case 'HORIZONTAL_RULE':
        $("#content").append("<hr style='display: flex;'>");
        break;
      default:
        console.log("Invalid attribute " + name);
    }
  });
}

function parseGroup(group) {
  var width = 100 / group.length;
  $.each(group, function(name, content) {
    switch(content.type) {
      case 'GAUGE':
        insertGauge(content, width);
        break;
      case 'BAR':
        insertBar(content, width);
        break;
      case 'EMPTY':
        $("#content").append("<div style='width:" + width + "%; float: left;'></div>");
        break;
      default:
        console.log("Invalid attribute " + name);
    }
  });
}

function insertText(textElement) {
  var headingId = "heading" + headingCounter ;

  if(textElement.id){
    headingId = textElement.id;
  }

  switch(textElement.style) {
    case 'REPORT_TITLE':
      $("#content").append("<span id='reportTitle' class='title-blue'>" + escapeHtml(textElement.content) + "</span>");
      break;
    case 'H1':
      if (textElement.id) {
        $("#content").append(MeerkatHelper.Util.insertTitle(headingId, "h4", textElement.content, textElement.description, INFO_IMAGE, counter));
        $("#"+headingId).append("<a href='#reportTitle'><img title='Back to top' src='" + BACKTOTOP_IMG +"' width='18' height='18' style='margin-left: 8px; margin-top: 12px; position: relative;'></img></a>");
      }
      else {
        $("#content").append(MeerkatHelper.Util.insertTitle(headingId, "h1", textElement.content, textElement.description, INFO_IMAGE, counter));
      }
      headingCounter++;
      counter++;
      break;
    case 'H1_HIDDEN':
      $("#content").append(MeerkatHelper.Util.insertTitle(headingId, "h1", textElement.content, textElement.description, SPANNER_IMG, counter));
      headingCounter++;
      counter++;
      break;
    case 'H2_HIDDEN':
      $("#content").append(MeerkatHelper.Util.insertTitle(headingId, "h2", textElement.content, textElement.description, INFO_IMAGE, counter));
      headingCounter++;
      counter++;
      break;
    case 'H3_HIDDEN':
      $("#content").append(MeerkatHelper.Util.insertTitle(headingId, "h3", textElement.content, textElement.description, INFO_IMAGE, counter));
      headingCounter++;
      counter++;
      break;
    case 'DESCRIPTION':
      $("#content").append("<p id='text" + textCounter + "' name='" + textElement.seleniumName + "'>" + escapeHtml(textElement.content) + "</p>");
      textCounter++;
      break;
    case 'ADVICE':
      $("#content").append(MeerkatHelper.Util.insertAdvice(adviceCounter, textElement.content, textElement.link, "nature-ADVICE"));
      adviceCounter++;
      break;
    case 'WARNING':
      $("#content").append(MeerkatHelper.Util.insertAdvice(adviceCounter, textElement.content, textElement.link, "nature-WARN"));
      adviceCounter++;
      break;
    default:
      console.log("Unknown text style " + textElement.style);
  }
}

function getXData(xdataStart, xdataLength, xdataStep) {
  for (var i = 0; i < xdataCache.length; i++) {
    if (xdataCache[i].xdataStart === xdataStart) {
      if (xdataCache[i].xdataLength === xdataLength) {
        if (xdataCache[i].xdataStep === xdataStep) {
          return xdataCache[i].xdata;
        }
      }
    }
  }
  var xdata = [];
  // For performance, array is grown to required size first
  // length - 1 as data is populated from index = 0 and then unshifted after with 'x' as first element
  xdata[xdataLength - 1] = 0;
  var accumulator = xdataStart;
  for (var x = 0; x < xdata.length; x++) {
    xdata[x] = accumulator;
    accumulator = accumulator + xdataStep;
  }
  // TODO: Why is performance much better with unshift after the array has been populated?
  xdata.unshift('x');
  var dataElement = {xdataStart : xdataStart, xdataLength : xdataLength, xdataStep : xdataStep, xdata : xdata};
  xdataCache.push(dataElement);
  return xdata;
}

function escapeHtml(unsafe) {
  if (unsafe == undefined) {
    return undefined;
  }
  return unsafe
       .replace(/&/g, "&amp;")
       .replace(/</g, "&lt;")
       .replace(/>/g, "&gt;")
       .replace(/"/g, "&quot;")
       .replace(/'/g, "&#039;");
}

function ToggleVisibility(divClass)
{
  var els = document.getElementsByClassName(divClass);
  for(var i = 0; i < els.length; i++)
  {
    els[i].style.display = els[i].style.display == "none" ? "block" : "none";
  }
}

function insertGauge(graph, width) {
  if (graph.linkId) {
    $("#content").append(MeerkatHelper.Util.insertChartDivWithLink(width, counter, graph.linkTitle, graph.linkId));
  }
  else {
    $("#content").append(MeerkatHelper.Util.insertChartDiv(width, counter));
  }

  var chart = bb.generate({
    bindto: "#graph" + counter,
    title: {
      text: (graph.linkId)? "" : graph.name,
      position: "center"
    },
    data: {
      columns: [
        [graph.name, 0]
      ],
      type: 'gauge',
    },
    padding: {
      left: 20,
      right: 20
    },
    gauge: {
      label: {
        format: function(value, ratio) {
          return value.toFixed(1) + (graph.unit == "PERCENTAGE" ? '%' : ' ' + graph.unit);
        }
      },
      max: graph.max, // 100 is default
    },
    legend: {
      hide: true,
    },
    tooltip: {
      format: {
        value: function(value, ratio) {
          return value.toFixed(1) + (graph.unit == "PERCENTAGE" ? '%' : ' ' + graph.unit);
        }
      }
    },
    size: {
      height: (graph.linkId)? 150 : 200
    }
  });
  setTimeout(function () {
    chart.load({
      columns: [[graph.name, graph.value]]
    });
  }, 100);
  counter++;
}

function insertBar(graph, width) {
  $("#content").append(MeerkatHelper.Util.insertChartDiv(width, counter));

  var emptyData = graph.names.reduce(function (result, x, i) {
    if (graph.values[i] != 0) {
      result.push([x, 0]);
    }
    return result;
  }, []);

  var data = graph.names.reduce(function (result, x, i) {
    if (graph.values[i] != 0) {
      result.push([x, graph.values[i]]);
    }
    return result;
  }, []);

  var chart = bb.generate({
    bindto: "#graph" + counter,
    title: {
      text: graph.chartTitle,
      position: "center"
    },
    padding: {
      left: 30,
      right: 30
    },
    data: {
      columns: emptyData,
      colors: {
        CPU: '#ff6b00',
        "Non-fragment" : '#ffc700',
        Fragment: '#00c1de',
        VSYNC: '#95d600',
        Unknown: '#7d868c'
      },
      labels: {
        colors: "black",
        format: function(value, ratio) {
          if (value > graph.max - (graph.max / 10)) {
            return "";
          }
          return value.toFixed(1) + (graph.unit == "PERCENTAGE" ? '%' : '');
        }
      },
      type: "bar"
    },
    bar: {
      width: {
        max: 70
      }
    },
    axis: {
      x: {
        tick: {
          values: []
        }
      },
      y: {
        max: graph.max,
        padding: 0
      }
    },
    tooltip: {
      show: true,
      doNotHide: false,
      format: {
        title: function(x) { return ""; },
        value: function(value, ratio, id, index) { return value.toFixed(1) + (graph.unit == "PERCENTAGE" ? '%' : ''); }
      },
      contents: {
        bindto: '#tooltip'
      },
    },
    size: {
      height: 200
    }
  });
  setTimeout(function () {
    chart.load({
      columns: data
    });
  }, 100);
  counter++;
}

function insertDualAxis(graph, width) {
  $("#content").append(MeerkatHelper.Util.insertChartDiv(width, counter));

  var xdata = getXData(graph.xdataStart, graph.xdataLength, graph.xdataStep);
  graph.y2data.unshift('dualAxisY2');

  var columnData = [xdata, graph.y2data];

  for (let i = 0; i < graph.ydata.length; i++) {
    graph.ydata[i].unshift(graph.ylabel[i]);
    columnData.push(graph.ydata[i]);
  }

  let colourData = new Object();
  colourData['dualAxisY2'] = graph.y2colour
  for (let i = 0; i < graph.ylabel.length; i++) {
    colourData[graph.ylabel[i]] = graph.ycolours[i];
  }

  if (graph.threshold.enabled) {
    colourData['threshold'] = 'green';
    let data = new Array(graph.xdataLength).fill(graph.threshold.value);
    data.unshift("threshold");
    columnData.push(data);
  }

  var chart = bb.generate({
    bindto: "#graph" + counter,

    padding: {
      top: 8,
      right: 70,
      bottom: 10
    },
    data: {
      x: 'x',
      columns: columnData,
      axes: {
        dualAxisY2: 'y2'
      },
      names: {
        dualAxisY2: graph.y2axislabel,
        threshold: graph.threshold.label
      },
      colors: colourData
    },
    point: {
      focus: {
        only: true
      }
    },
    axis: {
      x: {
        padding: {left: 0, right: 0},
        label: {
          text: graph.xlabel
        },
        type: 'timeseries',
        tick: {
          format: MeerkatHelper.Util.msToTime,
          fit: false
        }
      },
      y: {
        padding: {top: 0, bottom: 0},
        min: 0, // Required so all labels are visible and for correct axis position
        max: graph.ymax,
        label: {
          text: graph.yaxislabel,
          position: 'outer-middle'
        }
      },
      y2: {
        show: true,
        padding: {top: 0, bottom: 0},
        min:0, // See comment above.
        max: graph.y2max,
        tick: {
          count: graph.y2tickcount,
            format: function(value) {
              // Display whole number (0 decimal places)
              return value.toFixed(0);
            }
        },
        label: {
          text: graph.y2axislabel,
          position: 'outer-middle'
        }
      }
    },
    zoom: {
      enabled: true,
      type: "drag",
      onzoomend: function(e) {
        if (chart.zoomStart == undefined ||(chart.zoomStart != e[0] && chart.zoomEnd != e[1])) {
          chart.zoomStart = e[0];
          chart.zoomEnd = e[1];

          var frames = MeerkatHelper.Util.timesToFrameRange(e[0], e[1], binToFrameMapper);
          $('[class="RegionLegend ' + graph.group + '"]').each(function(i, el) {
            $(el).text("Showing frames " + frames);
          });

          // Zoom all charts to this level
          for (let i = 0; i < chartGroups[graph.group].length; i++) {
            chartGroups[graph.group][i].zoom([e[0], e[1]]);
          }
        }
      },
      resetButton: false
    },
    tooltip: {
      linked: {
        name: graph.group
      },
      format: {
        value: function (value, ratio, id) {
          var format = id === 'dualAxisY2' ? d3.format('.1f') : d3.format('.2f');
          return format(value);
        },
        name: function(name) {
          if(graph.scaleLabel === "" || name == graph.y2axislabel) {
            return name;
          } else {
            return name + " (" + graph.scaleLabel + ")";
          }
        },
        title: function(x) {
          // Workaround to stop d3 converting x to a date (since it is a timeseries)
          var clearFormat = d3.format("");
          var ms = clearFormat(x);
          var time = MeerkatHelper.Util.msToTime(ms);
          if (typeof(binToFrameMapper) !== 'undefined') {
            var frameRange = MeerkatHelper.Util.timeToFrameRange(ms, binToFrameMapper);
            return time + " " + frameRange;
          } else {
            return time;
          }
        },
      }
    },
    grid: {
      x: {
        show: true
      },
      y: {
        show: true
      }
    },
    size: {
      height: 250
    },
    onafterinit: function() {
      this.$.main.on("click", function() {
        chart.zoomStart = undefined;
        chart.zoomEnd = undefined;

        $('[class="RegionLegend ' + graph.group + '"]').each(function(j, el) {
          $(el).text("");
        });

        // Unzoom all charts
        for (var i = 0; i < chartGroups[graph.group].length; i++) {
          chartGroups[graph.group][i].unzoom();
        }
      });
    }
  });

  // Add frame selection info
  $("#content").append(`
  <center class="interactiveLegend">
  <div class="Row">
  <p style='float: left; margin-left: 0; margin-top: -37px;' class="RegionLegend ` + graph.group + `"></p>
  </div>
  </center>
  `);

  if (chartGroups[graph.group] == undefined) {
    chartGroups[graph.group] = [];
  }
  chartGroups[graph.group].push(chart);

  counter++;
}

function insertInteractive(graph) {
  $("#content").append(MeerkatHelper.Util.insertChartDiv(false, counter));

  var xdata = getXData(graph.xdataStart, graph.xdataLength, graph.xdataStep);
  graph.ydata.unshift(graph.ylabel);

  graph.boundness.forEach( function(data) {
    data['class'] = data['type'];
    delete data['type'];
  });

  // Setup screenshot data
  var lineArray = [];
  var screenshotLen = 0;
  if (graph.screenshots != undefined) {
    screenshotLen = graph.screenshots.length;
  }
  for (let i = 0; i < screenshotLen; i++) {
    lineArray.push({
        value: graph.screenshots[i].timeInMs,
        text: "S",
        class: "screenshot",
    });
  }

  var screenshotArray = [];
  var screenshotPathList = []
  var currentImageIndex = -1;
  var currentFrameTitle = "";
  var len = xdata.length;
  for (let i = 0; i < len; i++) {
    screenshotArray.push("");
    screenshotPathList.push("");
  }
  for (let i = 0; i < screenshotLen; i++) {
    var index = graph.screenshotBins[i];
    var image = '<img src="data:image/png;base64,' + graph.screenshots[i].base64
      + '" height="' + graph.screenshots[i].height + '" vspace="5"/>'
      + '<p style="text-align:center;background-color: #f5f5f5;"> Use middle click to enlarge image &#128269 </p>';
    screenshotArray[index] = image;

    screenshotPathList[index] = graph.screenshotPathList[i] ;

    if (index - 1 > 0) {
      screenshotArray[index - 1] = image;
      screenshotPathList[index - 1] = screenshotPathList[index];
    }

    if (index + 1 < len) {
      screenshotArray[index + 1] = image;
      screenshotPathList[index + 1] = screenshotPathList[index];
    }
  }

  var chart = bb.generate({
    bindto: "#graph" + counter,
    padding: {
      top: 8,
      right: 50,
      bottom: 10
    },
    data: {
      x: 'x',
      columns: [
        xdata,
        graph.ydata
      ]
    },
    regions: graph.boundness,
    point: {
      focus: {
        only: true
      }
    },
    zoom: {
      enabled: true,
      type: "drag",
      onzoomend: function(e) {
        if (chart.zoomStart == undefined ||(chart.zoomStart != e[0] && chart.zoomEnd != e[1])) {
          chart.zoomStart = e[0];
          chart.zoomEnd = e[1];

          var frames = MeerkatHelper.Util.timesToFrameRange(e[0], e[1], binToFrameMapper);
          $('[class="RegionLegend ' + graph.group + '"]').each(function(i, el) {
            $(el).text("Showing frames " + frames);
          });

          // Zoom all charts to this level
          for (let i = 0; i < chartGroups[graph.group].length; i++) {
            chartGroups[graph.group][i].zoom([e[0], e[1]]);
          }
        }
      },
      resetButton: false
    },
    axis: {
      x: {
        padding: {left: 0, right: 0},
        label: {
          text: "Time (s)"
        },
        type: 'timeseries',
        tick: {
          format: MeerkatHelper.Util.msToTime,
          fit: false
        }
      },
      y: {
        padding: {top: 0, bottom: 0},
        min: 0, // Required so all labels are visible and for correct axis position
        max: graph.ymax,
        label: {
          text: graph.ylabel,
          position: 'outer-middle'
        }
      }
    },
    grid: {
      x: {
        lines: lineArray
      }
    },
    tooltip: {
      doNotHide: false,
      linked: {
        name: graph.group
      },
      format: {
        title: function(x) {
          // Workaround to stop d3 coverting x to a date (since it is a timeseries)
          var clearFormat = d3.format("");
          var ms = clearFormat(x);
          var frameRange = MeerkatHelper.Util.timeToFrameRange(ms, binToFrameMapper);
          var time = MeerkatHelper.Util.msToTime(ms);
          currentFrameTitle = time + " " + frameRange;
          return currentFrameTitle;
        },
        value: function (value, ratio, id) {
          var format = id === 'FPS' ? d3.format('.1f') : d3.format('.2f');
          return format(value);
        }
      },
      onshown: function(selectedData) {
        $(this.$.tooltip._groups[0][0]).append(screenshotArray[selectedData[0].index]);
        currentImageIndex = selectedData[0].index;
      }
    },
    legend: {
      show: false
    },
    size: {
      height: 250
    },
    onafterinit: function() {
      this.$.main.on("click", function() {
        chart.zoomStart = undefined;
        chart.zoomEnd = undefined;

        $('[class="RegionLegend ' + graph.group + '"]').each(function(i, el) {
          $(el).text("");
        });

        // Unzoom all charts
        for (let i = 0; i < chartGroups[graph.group].length; i++) {
          chartGroups[graph.group][i].unzoom();
        }
      });

      const handle_click = () => {
        currentImagePath = screenshotPathList[currentImageIndex];
        if (currentImagePath != "") {
          var image = new Image();
          var image_filename = currentImagePath.replaceAll("\\","/").split("/").reverse()[0];
          var images_dir = currentImagePath.replaceAll("\\","/").split("/").reverse()[1];
          image.onload = function () {
            showFullResolutionImage(currentImagePath);
          }
          image.onerror = function () {
            alert("The high resolution image (" + image_filename +") for this frame can not be found.\n" +
              "Ensure that the '" + images_dir +"' directory is located in the same directory as the Performance Advisor report.");
          }
          image.src = currentImagePath;
        }
      }

      this_div = document.getElementById("graph" + counter);
      this_div.addEventListener("mousedown", function (event) {
        //Check if this is a middle click
        if (event.which == 2)
          handle_click();
      }
      );
    },
    plugins: [
      new bb.plugin.calipers({calipers: graph.regions})
    ]
  });

  if (chartGroups[graph.group] == undefined) {
    chartGroups[graph.group] = [];
  }
  chartGroups[graph.group].push(chart);

  legend = `
  <center class="interactiveLegend">
  <div class="Row" style="margin-top: -8px;">`;
  for (let key in graph.boundnessTypesWithCssRules) {
      // Don't show unknown in the legend
      if(key == "Unknown") {
        continue;
      }
      legend += `<div class="Column">
      <div class="box ${graph.boundnessTypesWithCssRules[key]}"></div>
      <div class="RegionLegend">${key} bound</div>
      </div>`;
  }
  legend += `
  </div>
  <div class="Row">
  <p style='float:left; margin-left: 0; margin-top: -37px;' class="RegionLegend ` + graph.group + `"></p>
  </div>
  </center>
  `;
  $("#graph" + counter).append(legend);

  counter++;
}
