#!/usr/bin/env bash
# This confidential and proprietary software may be used only as
# authorised by a licensing agreement from Arm Ltd.
#   Copyright: 2025 Arm Ltd. All Rights Reserved.
# The entire notice above must be reproduced on all authorised
# copies and copies may only be made to the extent permitted
# by a licensing agreement from Arm Ltd.

# Script to install Frame Advisor desktop icons on Linux.

cd $(dirname $0)

FA_BASENAME="FrameAdvisor-gui"

SCRIPT_DESCRIPTION="Syntax: $0 [frame-advisor-executable | --help]

Installs Frame Advisor's icons. If Frame Advisor cannot be found
automatically, the first parameter should give the path to Frame
Advisor's main executable ($FA_BASENAME)."

if ! which xdg-icon-resource > /dev/null
then
    echo "ERROR: This script requires xdg-icon-resource to be installed." >&2
    echo "This is part of xdg-utils. Install xdg-utils and try again." >&2
    exit 1
fi

if [ $# -eq 1 ]
then
    if [ "$1" == "--help" ]
    then
        echo "$SCRIPT_DESCRIPTION"
        exit 0
    fi

    FA_FULL_PATH=$(realpath "$1")
else
    FA_FULL_PATH=$(realpath "..")
fi

# Directory path passed rather than path to Frame Advisor executable
if [ -d "$FA_FULL_PATH" ]
then
    FA_FULL_PATH=$FA_FULL_PATH/$FA_BASENAME
fi

if [ ! -f "$FA_FULL_PATH" ]
then
    echo "ERROR: Can't find Frame Advisor executable at $FA_FULL_PATH" >&2
    echo "$SCRIPT_DESCRIPTION"
    exit 1
fi

# If the passed path isn't executable, the user has probably referenced
# the wrong file.
if [ ! -x "$FA_FULL_PATH" ]
then
    echo "ERROR: Frame Advisor executable $FA_FULL_PATH is not marked executable" >&2
    echo "$SCRIPT_DESCRIPTION"
    exit 1
fi

echo "Found Frame Advisor at $FA_FULL_PATH"

FRAME_ADVISOR_INSTALL_DIR=$(dirname "$FA_FULL_PATH")
DESKTOP_FILE_BASENAME="Frame Advisor.desktop"
MAIN_INSTALL_PATH="$HOME/.local/share/applications/$DESKTOP_FILE_BASENAME"
DESKTOP_FOLDER_INSTALL_PATH="$HOME/Desktop/$DESKTOP_FILE_BASENAME"
ICON_ID=FrameAdvisor-PS

echo "Installing Frame Advisor icon..."
xdg-icon-resource install --novendor --context apps --mode user --size 256 "${ICON_ID}.png"

echo "Creating Frame Advisor application launcher..."
sed "s!\\\${FRAME_ADVISOR_INSTALL_DIR}!${FRAME_ADVISOR_INSTALL_DIR}!" \
    "Frame Advisor.desktop.template" > "$MAIN_INSTALL_PATH"
chmod +x "$MAIN_INSTALL_PATH"

echo "Adding launcher to desktop..."
cp "$MAIN_INSTALL_PATH" "$DESKTOP_FOLDER_INSTALL_PATH"

# Set up some Gnome-specific settings.
# (Gnome is the default desktop environment for Ubuntu.)
if which gio > /dev/null
then
    # Gnome requires a user to select “Allow Launching” before it will allow a
    # launcher to work. Skip that step.
    echo "Enabling launcher..."
    gio set "$DESKTOP_FOLDER_INSTALL_PATH" metadata::trusted true

    # Set a custom icon for the main FA executable
    echo "Setting custom icon in installation directory..."
    gio set -t unset "$FA_FULL_PATH" metadata::custom-icon
    gio set -t string "$FA_FULL_PATH" metadata::custom-icon-name "$ICON_ID"
fi

echo "Frame Advisor icons installed."
exit 0
