/**
 * Copyright (C) 2016-2025 by Arm Limited (or its affiliates). All rights reserved.
 * Use, modification and redistribution of this file is subject to your possession of a
 * valid End User License Agreement for the Arm Product of which these examples are part of
 * and your compliance with all applicable terms and conditions of such license agreement.
 *
 * Redistribution permitted only in object code form and only as
 * part of Software Applications developed by you or your permitted users. If you
 * choose to redistribute the whole or any part of Example Code, you agree: (a)
 * to ensure that they are licensed for use only as part of Software Applications
 * and only for execution on microprocessors manufactured or simulated under
 * licence from Arm; (b) not to use Arm's or any of its licensors names, logos or
 * trademarks to market Software Applications; (c) to include valid copyright
 * notices on Software Applications, and preserve any copyright notices which are
 * included with, or in, the Example Code; and (d) to ensure that any further
 * redistribution is limited to redistribution by either or both your customers
 * and your authorised distributors as part of Software Applications and that
 * your customers and your authorised distributors comply with these terms.
 */
/** @file */

#ifndef INCLUDE_INTRINSICS
#define INCLUDE_INTRINSICS

/**
 * @defgroup    intrinsics   Aliases for intrinsic assembler operations
 * @{
 *
 * @def         STRINGIFY_TOK
 * @brief       The parameter token(s) are converted into a string
 */
#define STRINGIFY_TOK(x) #x

/**
 * @def         STRINGIFY_VAL
 * @brief       The value of the parameter is converted into a string
 */
#define STRINGIFY_VAL(x) STRINGIFY_TOK(x)

#define MEMORY_MAPPED_REGISTER_32(address) (*((volatile unsigned int *) (address)))

/*
 * @def         MRC(op0, op1, CRn, CRm, op2, Rt)
 * @brief       MRC instruction (or manually encoded MRS on AArch64)
 * @param       op0
 *              Is the op1/coproc parameter within the System register encoding space,
 *              in the range 0 to 7 or 8 to 15, respectively, encoded in the "op1"/"coproc" field.
 * @param       op1
 *              Is the op1/opc1 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op1"/"opc1" field.
 * @param       CRn
 *              Is the CRn parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRn" field.
 * @param       CRm
 *              Is the CRm parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRm" field.
 * @param       op2
 *              Is the op2/opc2 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op2"/"opc2" field.
 * @param       Rt Type "unsigned int"
 *              L-value to write the result into.
 *
 * @def         MCR(op0, op1, CRn, CRm, op2, Rt)
 * @brief       MCR instruction (or manually encoded MSR on AArch64)
 * @param       op0
 *              Is the op1/coproc parameter within the System register encoding space,
 *              in the range 0 to 7 or 8 to 15, respectively, encoded in the "op1"/"coproc" field.
 * @param       op1
 *              Is the op1/opc1 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op1"/"opc1" field.
 * @param       CRn
 *              Is the CRn parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRn" field.
 * @param       CRm
 *              Is the CRm parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRm" field.
 * @param       op2
 *              Is the op2/opc2 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op2"/"opc2" field.
 * @param       Rt Type "unsigned int"
 *              Value to write into the register.
 *
 * @def         MRS(name, Rt)
 * @brief       MRS instruction
 * @param       name
 *              Name of the special purpose register to read.
 * @param       Rt Type "unsigned int"
 *              L-value to write the result into.
 *
 * @def         MSR(name, Rt)
 * @brief       MSR instruction
 * @param       name
 *              Name of the special purpose register to write into.
 * @param       in Type "unsigned int"
 *              Value to write into the register.
 *
 * @def         MRRC(op0, op1, CRm, Rt)
 * @brief       MRRC instruction
 * @param       op0
 *              Is the System register encoding space,
 *              in the range 8 to 15, encoded in the "op1"/"coproc" field.
 * @param       op1
 *              Is the op1/opc1 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op1"/"opc1" field.
 * @param       CRm
 *              Is the CRm parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRm" field.
 * @param       Rt Type "unsigned int"
 *              L-value to write the first result into.
 * @param       Rt2 Type "unsigned int"
 *              L-value to write the second result into.
 *
 * @def         MCRR(op0, op1, CRm, Rt)
 * @brief       MCRR instruction
 * @param       op0
 *              Is the System register encoding space,
 *              in the range 8 to 15, encoded in the "op1"/"coproc" field.
 * @param       op1
 *              Is the op1/opc1 parameter within the System register encoding space,
 *              in the range 0 to 7, encoded in the "op1"/"opc1" field.
 * @param       CRm
 *              Is the CRm parameter within the System register encoding space,
 *              in the range 0 to 15, encoded in the "CRm" field.
 * @param       Rt Type "unsigned int"
 *              First value to write into the register.
 * @param       Rt2 Type "unsigned int"
 *              Second value to write into the register.
 */
#if defined(__ARMCC_VERSION) && (__ARMCC_VERSION < 6000000)
#  define  MRC(op0, op1, CRn, CRm, op2, Rt)    __asm ("mrc p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", " STRINGIFY_VAL(Rt) ", c" STRINGIFY_VAL(CRn) ", c" STRINGIFY_VAL(CRm) ", " STRINGIFY_VAL(op2))
#  define  MCR(op0, op1, CRn, CRm, op2, Rt)    __asm ("mcr p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", " STRINGIFY_VAL(Rt) ", c" STRINGIFY_VAL(CRn) ", c" STRINGIFY_VAL(CRm) ", " STRINGIFY_VAL(op2))
#  define  MRRC(op0, op1, CRm, Rt, Rt2)        __asm ("mrrc p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", " STRINGIFY_VAL(Rt) ", " STRINGIFY_VAL(Rt2) ", c" STRINGIFY_VAL(CRm))
#  define  MCRR(op0, op1, CRm, Rt, Rt2)        __asm ("mcrr p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", " STRINGIFY_VAL(Rt) ", " STRINGIFY_VAL(Rt2) ", c" STRINGIFY_VAL(CRm))
#  define  MRS(name, Rt)                       __asm ("mrs " STRINGIFY_VAL(Rt) ", " STRINGIFY_VAL(name))
#  define  MSR(name, Rt)                       __asm ("msr " STRINGIFY_VAL(name) ", " STRINGIFY_VAL(Rt))
#else
#  define  MRC(op0, op1, CRn, CRm, op2, Rt)    asm volatile ("mrc p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", %0, c" STRINGIFY_VAL(CRn) ", c" STRINGIFY_VAL(CRm) ", " STRINGIFY_VAL(op2) : "=r"(Rt))
#  define  MCR(op0, op1, CRn, CRm, op2, Rt)    asm volatile ("mcr p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", %0, c" STRINGIFY_VAL(CRn) ", c" STRINGIFY_VAL(CRm) ", " STRINGIFY_VAL(op2) :: "r"(Rt))
#  define  MRRC(op0, op1, CRm, Rt, Rt2)        asm volatile ("mrrc p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", %0, %1, c" STRINGIFY_VAL(CRm) : "=r"(Rt), "=r"(Rt2))
#  define  MCRR(op0, op1, CRm, Rt, Rt2)        asm volatile ("mcrr p" STRINGIFY_VAL(op0) ", " STRINGIFY_VAL(op1) ", %0, %1, c" STRINGIFY_VAL(CRm) :: "r"(Rt), "r"(Rt2))
#  define  MRS(name, Rt)                       asm volatile ("mrs %0, " STRINGIFY_VAL(name) : "=r"(Rt))
#  define  MSR(name, Rt)                       asm volatile ("msr " STRINGIFY_VAL(name) ", %0" :: "r"(Rt))
#endif

/** @} */

#endif /* INCLUDE_INTRINSICS */
