import {
	select as d3Select
} from "d3-selection";
import {loadConfig} from "../../config/config";
import Plugin from "../Plugin";
import Options from "./Options";
import {callFn} from "../../module/util";

/**
 * Calipers plugin<br>
 * - **NOTE:**
 *   - Plugins aren't built-in. Need to be loaded or imported to be used.
 *   - Non required modules from billboard.js core, need to be installed separately.
 * - **Required modules:**
 *   - [d3-selection](https://github.com/d3/d3-selection)
 * @class plugin-calipers
 * @requires d3-selection
 * @param {Object} options Calipers plugin options
 * @extends Plugin
 * @return {Calipers}
 */
export default class Calipers extends Plugin {
	public config: any;
	public parentNode: any;

  constructor(options) {
    super(options);
    this.config = new Options();

    return this;
  }

  $init() {
    const {$$} = this;
    loadConfig.call(this, this.options);

    const calipers = this.config.calipers;

    // Find largest caliper level
    var largest = 0;
    for (let i = 0; i < calipers.length; ++i) {
      if (calipers[i].stackLevel > largest) {
          largest = calipers[i].stackLevel;
      }
    }
    largest = (largest + 1) * 20;

    if (calipers.length > 0) {
      $$.config.padding_top = largest - 10;
      // Create a group for the calipers
      $$.$el.svg.append('g')
      .attr('class', 'calipers')
      .append('clipPath')
        .attr('id', 'calipers-clip')
        .append('rect')
          .attr('x', '0')
          .attr('y', '0')
          .attr('width', $$.state.width)
          .attr('height', largest + 1);
    }

    // For each caliper
    calipers.forEach(function(_d, i) {
      // Create a region
      $$.config.regions = $$.config.regions.concat({start: calipers[i].start, end: calipers[i].end, class: "caliper" + i + " nobg"});
      $$.$el.svg.select(".caliper" + i)
        .select('rect')
        .style('fill', 'none');

      // Get position
      var x = $$.scale.x(calipers[i].start);
      var x2 = $$.scale.x(calipers[i].end);

      // Get min/max x value
      var min = $$.scale.x.domain()[0];
      var max = $$.scale.x.domain()[1];

      // Work out if to slice text
      var text = calipers[i].name;
      var self = $$.$el.svg.append('text')
        .text(text);
      var textLength = self.node().getComputedTextLength();
      var add = false;
      while (textLength > (x2 - x)) {
        if (text.length == 0) {
          add = false;
          break;
        }
        add = true;
        text = text.slice(0, -1);
        self.text(text + '...');
        textLength = self.node().getComputedTextLength();
      }
      if (add) {
        text += "...";
      }
      self.remove();

      var group = $$.$el.svg.select('.calipers').append('g');
      // Add the main line
      group.append('line')
        .attr('x1', x)
        .attr('x2', x2)
        .attr('y1', (calipers[i].stackLevel * 20) + 15)
        .attr('y2', (calipers[i].stackLevel * 20) + 15)
        .attr('stroke-width', '2')
        .attr('stroke', 'grey !important')
        .attr("clip-path", "url(#calipers-clip)");
      // Add the name
      group.append('text')
        .text(text)
        .attr('id', 'caliper' + i)
        .attr('x', Math.round((x + x2) / 2))
        .attr('y', (calipers[i].stackLevel * 20) + 10)
        .attr('fill', 'black')
        .attr('fill-opacity', '1')
        .style('text-anchor', 'middle')
        .attr("clip-path", "url(#calipers-clip)");
      // Add the events
      group.append('rect')
        .attr('x', x)
        .attr('width', x2 - x)
        .attr('y', (calipers[i].stackLevel * 20))
        .attr('height', 15)
        .attr('fill', "transparent")
        .attr("clip-path", "url(#calipers-clip)")
        .style('cursor', 'pointer')
        .on('mouseover', function(_d, _i) {
          let group = d3Select(this)
          .select(function(){
            return this.parentNode;
          });
          group.select('text')
          .style('font-weight', 'bold');
          $$.$el.svg.select("." + group.select('text').attr('id'))
            .select('rect')
            .style('fill', 'rgba(0, 0, 0, 0.2)')
            .style('stroke', 'rgba(0, 0, 0, 0.6)')
            .style('stroke-width', '2')
        })
        .on('mouseout', function(_d, _i) {
          let group = d3Select(this)
          .select(function(){
            return this.parentNode;
          });
          group.select('text')
          .style('font-weight', 'normal');
          $$.$el.svg.select("." + group.select('text').attr('id'))
            .select('rect')
            .style('fill', 'none')
            .style('stroke', 'none')
            .style('stroke-width', '0')
        })
        .on('click', function(_d, _i) {
          let group = d3Select(this)
          .select(function(){
            return this.parentNode;
          });
          let j = group.select('text').attr('id').replace('caliper','');
          let start = calipers[j].start;
          let end = calipers[j].end;

          if (start < min) {
            start = min;
          }

          if (end > max) {
            end = max;
          }

          $$.api.zoom([start, end]);
          callFn($$.config.zoom_onzoomend, $$.api, [start, end]);
        });
    });
  }

  $redraw() {
    const $$ = this.$$;
    const calipers = this.config.calipers;

    // Get transform value
    var transform = $$.$el.svg.select('g').attr('transform');
    transform = transform.replace('translate(','');
    transform = transform.split(',')[0];

    // Update transform/clip path
    $$.$el.svg.select('.calipers')
      .attr('transform','translate(' + transform + ', 0)')
      .select('clipPath')
        .select('rect')
          .attr('width', $$.state.width);

    // For each caliper
    $$.$el.svg.select('.calipers').selectAll('g').each(function(_d, i) {
      // Get region position
      var x = $$.scale.x(calipers[i].start);
      var x2 = $$.scale.x(calipers[i].end);

      var textX = x;
      var textX2 = x2;

      if (textX < 0 && textX2 > 0) {
        textX = 0;
      }

      if (textX2 > $$.state.width && textX < $$.state.width) {
        textX2 = $$.state.width;
      }

      // Work out if to slice text
      var text = calipers[i].name;
      var self = $$.$el.svg.append('text')
        .text(text);
      var textLength = self.node().getComputedTextLength();
      var add = false;
      while (textLength > (textX2 - textX)) {
        if (text.length == 0) {
          add = false;
          break;
        }
        add = true;
        text = text.slice(0, -1);
        self.text(text + '...');
        textLength = self.node().getComputedTextLength();
      }
      if (add) {
        text += "...";
      }
      self.remove();

      d3Select(this)
      // Move the main line
      .select('line')
      .transition()
      .attr('x1', x)
      .attr('x2', x2)
      // Move the name
      .select(function(){
          return (<HTMLElement> this).parentElement;
      })
      .select('text')
      .transition()
      .text(text)
      .attr('x', Math.round((textX + textX2) / 2))
      // Move the rect
      .select(function(){
          return (<HTMLElement> this).parentElement;
      })
      .select('rect')
      .attr('x', x)
      .attr('width', x2 - x)
    });
  }
}
