/*
 * Decompiled with CFR 0.152.
 */
package com.arm.streamline.performanceadvisor.analyser;

import com.arm.streamline.performanceadvisor.Application;
import com.arm.streamline.performanceadvisor.analyser.PerFrameChartData;
import com.arm.streamline.performanceadvisor.analyser.StandardPerFrameAnalyser;
import com.arm.streamline.performanceadvisor.capturedata.BoundnessProvider;
import com.arm.streamline.performanceadvisor.capturedata.BoundnessType;
import com.arm.streamline.performanceadvisor.capturedata.CaptureData;
import com.arm.streamline.performanceadvisor.capturedata.CpuProvider;
import com.arm.streamline.performanceadvisor.capturedata.Region;
import com.arm.streamline.performanceadvisor.common.FatalError;
import com.arm.streamline.performanceadvisor.common.ProblemList;
import com.arm.streamline.performanceadvisor.common.ResourceProblem;
import com.arm.streamline.performanceadvisor.counterseries.ArraySeries;
import com.arm.streamline.performanceadvisor.counterseries.CounterSeries;
import com.arm.streamline.performanceadvisor.counterseries.ExpressionSeries;
import com.arm.streamline.performanceadvisor.counterseries.SelectOneSeries;
import com.arm.streamline.performanceadvisor.counterseries.WildcardExpressionSeries;
import com.arm.streamline.performanceadvisor.report.SectionDefinition;
import com.arm.streamline.performanceadvisor.report.SeriesDefinition;
import com.arm.streamline.performanceadvisor.section.Element;
import com.arm.streamline.performanceadvisor.section.TextBlock;
import com.arm.streamline.performanceadvisor.section.TextStyle;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import org.eclipse.jdt.annotation.NonNull;

public class ExpressionPerFrameAnalyser
extends StandardPerFrameAnalyser {
    private static final String SERIES_NAME_ERROR = "All series in report configuration must have a name";
    private static final int NUM_FRAMES_THRESHOLD = 10;
    private static final double FRAME_PERCENTAGE_THRESHOLD = 20.0;
    private static final double OVERDRAW_THRESHOLD = 2.5;
    private static final String BASIC_OVERDRAW_ADVICE = "Your application has areas of high overdraw.";
    private static final String REGION_OVERDRAW_ADVICE = "This region of your application has areas of high overdraw.";
    private static final String FRAGMENT_OVERDRAW_ADVICE = "Areas with high levels of overdraw may cause your application to become fragment bound. Reduce or avoid overdraw in these regions to increase performance";
    private static final String ARITHMETIC_ADVICE = "To prevent the GPU spending time processing arithmetic operations, reduce computation in your shaders.";
    private static final String TEXTURE_ADVICE = "To prevent the GPU spending time processing texture operations, reduce texture lookups.";
    private static final String VARYING_ADVICE = "To prevent the GPU spending time processing shader varying variables, reduce the data passed between the non-fragment and fragment shaders.";
    private static final String LOADSTORE__ADVICE = "The GPU seems to be busy with load / store operations.";
    private static final String CHART_UNDEFINED_ADVICE = "Ensure your JSON report configuration is formatted correctly, and all charts have data.";
    private static final String CHART_UNDEFINED_ERROR = "The chart definition provided to the analyser is not defined.";
    private static final String WILDCARD_SUFFIX = "*";
    private static final String EGL_WILDCARD = "$DrawCallsFrameEGLContext*";
    private static final String VULKAN_WILDCARD = "$DrawCallsFrameVkDevice*";
    private static final String UNKNOWN_TYPE_ERROR = "Type %s defined in config report does not exist";
    private static final String SERIES_EXPRESSION_ERROR = "Series %s in report config must contain a counter expression";
    private static final String SERIES_TITLE_ERROR = "All series in report config must have a title";
    public static final String OVERDRAW = "Overdraw";
    public static final String CPU_CYCLES = "CPU usage";
    private SectionDefinition sectionDefn;
    private String units;
    private Optional<String> description;
    private List<CounterSeries> seriesList;

    public ExpressionPerFrameAnalyser(CaptureData captureData, Optional<Double> threshold, SectionDefinition sectionDefn) {
        super(captureData, threshold.isPresent() ? threshold : sectionDefn.getThreshold());
        this.sectionDefn = sectionDefn;
        this.units = sectionDefn.getUnits();
        this.description = sectionDefn.getDescription();
        this.seriesList = new ArrayList<CounterSeries>();
        for (SeriesDefinition seriesData : sectionDefn.getSeries()) {
            try {
                CounterSeries series = this.getCounterSeriesFromData(seriesData);
                this.seriesList.add(series);
                this.addSeries(series);
            }
            catch (SectionDefinitionDataException e) {
                throw new FatalError("Problem with report configuration file: ", e);
            }
        }
    }

    public ExpressionPerFrameAnalyser(CaptureData captureData, SectionDefinition sectionDefn) {
        super(captureData, sectionDefn.getThreshold());
        this.sectionDefn = sectionDefn;
        this.units = sectionDefn.getUnits();
        this.description = sectionDefn.getDescription();
        this.seriesList = new ArrayList<CounterSeries>();
    }

    @Override
    public ProblemList initialise() {
        ProblemList problemList = new ProblemList();
        problemList.addAll(super.initialise());
        if (this.sectionDefn == null) {
            problemList.add(new ResourceProblem(ResourceProblem.Severity.ERROR, CHART_UNDEFINED_ERROR, CHART_UNDEFINED_ADVICE));
        }
        return problemList;
    }

    @Override
    public String getId() {
        return this.sectionDefn.getId();
    }

    @Override
    public String getYAxisLabel() {
        return this.units;
    }

    @Override
    public String getTitle() {
        if (this.sectionDefn.getTitle().isPresent()) {
            return this.sectionDefn.getTitle().get();
        }
        return "";
    }

    @Override
    public String getHelpText() {
        if (this.description.isPresent()) {
            return this.description.get();
        }
        return "";
    }

    public void addSeriesToList(CounterSeries series) {
        this.seriesList.add(series);
    }

    private CounterSeries getCounterSeriesFromData(SeriesDefinition seriesData) throws SectionDefinitionDataException {
        if (seriesData.getTitle().isBlank()) {
            throw new SectionDefinitionDataException(SERIES_TITLE_ERROR);
        }
        int currentBinData = this.captureData.getBinCount();
        CounterSeries counterSeries = switch (this.getChartType()) {
            case ChartType.PA_CPU_CYCLES -> this.handleCpuCyclesSeries();
            case ChartType.PA_OVERDRAW -> this.handleOverdrawSeries(seriesData);
            case ChartType.PA_DRAW_CALLS -> this.handleDrawCallsSeries(seriesData);
            default -> this.createExpressionSeries(seriesData);
        };
        assert (currentBinData == this.captureData.getBinCount());
        return counterSeries;
    }

    private @NonNull CounterSeries createExpressionSeries(SeriesDefinition seriesData) throws SectionDefinitionDataException {
        String name = seriesData.getTitle();
        if (name.isBlank()) {
            throw new SectionDefinitionDataException(SERIES_NAME_ERROR);
        }
        String expression = seriesData.getExpression();
        if (expression.isBlank()) {
            throw new SectionDefinitionDataException(String.format(SERIES_EXPRESSION_ERROR, seriesData.getTitle()));
        }
        boolean isWildcard = expression.trim().endsWith(WILDCARD_SUFFIX);
        if (isWildcard) {
            return new WildcardExpressionSeries(name, expression, this.units, seriesData.isAverage(), seriesData.isPerFrame(), seriesData.isMax(), seriesData.isMin(), seriesData.doNotShowIfAllZeroes());
        }
        return new ExpressionSeries(name, expression, this.units, seriesData.isAverage(), seriesData.isPerFrame(), seriesData.isMax(), seriesData.isMin(), seriesData.doNotShowIfAllZeroes());
    }

    private CounterSeries handleOverdrawSeries(SeriesDefinition seriesData) throws SectionDefinitionDataException {
        try {
            ArraySeries overdrawSeries;
            int captureBinCount = this.captureData.getBinCount();
            this.captureData.setBinSizeHundredMilSecAndWait();
            CounterSeries series = this.createExpressionSeries(seriesData);
            series.initialise(this.captureData);
            if (!series.isPresent()) {
                CounterSeries counterSeries = series;
                return counterSeries;
            }
            double[] data = series.toArray();
            this.captureData.setBinSizeOneMilSecAndWait();
            double[] overdrawCapture = new double[captureBinCount];
            int bin = 0;
            while (bin < captureBinCount) {
                double proportion = (double)bin / 100.0;
                int indexLeft = (int)Math.floor(proportion);
                int indexRight = indexLeft + 1;
                proportion -= (double)indexLeft;
                if (indexRight >= data.length) {
                    indexRight = data.length - 1;
                }
                overdrawCapture[bin] = (1.0 - proportion) * data[indexLeft] + proportion * data[indexRight];
                ++bin;
            }
            ArraySeries arraySeries = overdrawSeries = new ArraySeries(seriesData.getTitle(), overdrawCapture, seriesData.isPerFrame());
            return arraySeries;
        }
        finally {
            this.captureData.setBinSizeOneMilSecAndWait();
        }
    }

    private CounterSeries handleCpuCyclesSeries() {
        boolean hasMainThread;
        CpuProvider cpuProvider = this.captureData.getCpuProvider();
        CounterSeries cpuCyclesSeries = cpuProvider.getCyclesSeries();
        boolean bl = hasMainThread = cpuProvider.getMainThread() != null;
        if (hasMainThread || !cpuCyclesSeries.isPresent()) {
            Application.LOG.fine("CPU cycles not found. Not reporting CPU cycles per frame.");
        }
        return cpuCyclesSeries;
    }

    private CounterSeries handleDrawCallsSeries(SeriesDefinition seriesData) throws SectionDefinitionDataException {
        seriesData.setExpression(VULKAN_WILDCARD);
        CounterSeries vulkanSeries = this.createExpressionSeries(seriesData);
        seriesData.setExpression(EGL_WILDCARD);
        CounterSeries eglSeries = this.createExpressionSeries(seriesData);
        return new SelectOneSeries(seriesData.getTitle(), vulkanSeries, eglSeries);
    }

    private ChartType getChartType() {
        if (!this.sectionDefn.getType().isEmpty()) {
            String type = this.sectionDefn.getType().get();
            try {
                return ChartType.valueOf(type);
            }
            catch (IllegalArgumentException e) {
                Application.LOG.warning(String.format(UNKNOWN_TYPE_ERROR, type));
            }
        }
        return ChartType.DEFAULT;
    }

    @Override
    public List<Element> getAdvice(PerFrameChartData data, int userRegion) {
        ChartType type = this.getChartType();
        switch (type) {
            case PA_OVERDRAW: {
                return this.handleOverdrawAdvice(userRegion);
            }
        }
        if (this.sectionDefn.getTitle().get().contains("Shader")) {
            return this.handleShaderCoreAdvice(data, userRegion);
        }
        return Collections.emptyList();
    }

    private List<Element> handleOverdrawAdvice(int userRegion) {
        ArrayList<Element> advice = new ArrayList<Element>();
        Region region = this.captureData.getClippedRegionsProvider().getClippedRegion(userRegion);
        if (!this.seriesList.get(0).isPresent()) {
            return advice;
        }
        double[] overdrawCapture = this.seriesList.get(0).toArray();
        double[] overdrawRegion = Arrays.copyOfRange(overdrawCapture, region.getStart(), region.getEnd());
        int bin = 0;
        while (bin < overdrawRegion.length - 1) {
            if (overdrawRegion[bin] > 2.5) {
                try {
                    URI adviceUri = new URI(String.format("https://developer.arm.com/documentation/102643/latest/%s", "High-overdraw"));
                    if (this.captureData.getBoundnessProvider().getBoundnessForBinIndex(region.getStart() + bin).isBoundnessType(BoundnessType.FRAGMENT)) {
                        advice.add(new TextBlock(TextStyle.ADVICE, FRAGMENT_OVERDRAW_ADVICE, adviceUri.toURL()));
                        break;
                    }
                    if (userRegion != 0) {
                        advice.add(new TextBlock(TextStyle.ADVICE, REGION_OVERDRAW_ADVICE, adviceUri.toURL()));
                        break;
                    }
                    advice.add(new TextBlock(TextStyle.ADVICE, BASIC_OVERDRAW_ADVICE, adviceUri.toURL()));
                    break;
                }
                catch (URISyntaxException e) {
                    throw new FatalError("URI syntax error: ", e);
                }
                catch (MalformedURLException e) {
                    throw new FatalError("Malformed URL: ", e);
                }
            }
            ++bin;
        }
        return advice;
    }

    private List<Element> handleShaderCoreAdvice(PerFrameChartData data, int userRegion) {
        double percentage;
        ArrayList<Element> advice = new ArrayList<Element>();
        BoundnessProvider boundnessProvider = this.captureData.getBoundnessProvider();
        int gpuBoundCount = 0;
        int i = this.getStartIndex(userRegion);
        while (i < this.getEndIndex(userRegion)) {
            if (boundnessProvider.getBoundnessForBinIndex(i).isGpuBound()) {
                ++gpuBoundCount;
            }
            ++i;
        }
        int regionLength = this.getEndIndex(userRegion) - this.getStartIndex(userRegion);
        double d = percentage = regionLength > 0 ? (double)gpuBoundCount / (double)regionLength : 0.0;
        if (percentage > 20.0 || gpuBoundCount > 10) {
            int dominantSeriesIdx = data.findDominantSeries();
            CounterSeries dominantSeries = this.seriesList.get(dominantSeriesIdx);
            try {
                if (dominantSeries.getName().contains("Load/store")) {
                    URI adviceUri = new URI(String.format("https://developer.arm.com/documentation/102643/latest/%s", "High-load-store"));
                    advice.add(new TextBlock(TextStyle.ADVICE, LOADSTORE__ADVICE, adviceUri.toURL()));
                } else if (dominantSeries.getName().contains("Varying")) {
                    URI adviceUri = new URI(String.format("https://developer.arm.com/documentation/102643/latest/%s", "High-varying-load"));
                    advice.add(new TextBlock(TextStyle.ADVICE, VARYING_ADVICE, adviceUri.toURL()));
                } else if (dominantSeries.getName().contains("Texture")) {
                    URI adviceUri = new URI(String.format("https://developer.arm.com/documentation/102643/latest/%s", "High-texture-load"));
                    advice.add(new TextBlock(TextStyle.ADVICE, TEXTURE_ADVICE, adviceUri.toURL()));
                } else {
                    URI adviceUri = new URI(String.format("https://developer.arm.com/documentation/102643/latest/%s", "High-arithmetic-load"));
                    advice.add(new TextBlock(TextStyle.ADVICE, ARITHMETIC_ADVICE, adviceUri.toURL()));
                }
            }
            catch (MalformedURLException e) {
                throw new FatalError("Malformed URL: ", e);
            }
            catch (URISyntaxException e) {
                throw new FatalError("URI syntax error: ", e);
            }
        }
        return advice;
    }

    static enum ChartType {
        PA_CPU_CYCLES,
        PA_OVERDRAW,
        PA_DRAW_CALLS,
        DEFAULT;

    }

    public static class SectionDefinitionDataException
    extends Exception {
        public SectionDefinitionDataException(@NonNull String message) {
            super(message);
        }

        public SectionDefinitionDataException(@NonNull String message, Throwable t) {
            super(message, t);
        }
    }
}

