import sys
import csv
import json

import pandas as pd


def profile_csv_to_profile_dict(csv_path):
    profile_dict = {}
    try:
        csv_file = open(csv_path)
    except FileNotFoundError:
        print(f"Error opening file {csv_path}")
    else:
        with csv_file:
            rows = csv.DictReader(csv_file)
            for row in rows:
                key = row.pop("symbol")
                profile_dict[key] = row

    return profile_dict


def diff_dict_to_diff_df(diff_dict):
    diff_df = pd.DataFrame.from_dict(diff_dict, 'index')
    diff_df = diff_df[['Rank'] +
                      [col for col in diff_df.columns if col != 'Rank']]

    return diff_df


def diff_profiles(curr_profile, prev_profile):
    diff = {}
    new_funcs = []
    removed_funcs = []
    for function in curr_profile:
        if function not in prev_profile:
            new_funcs.append(function)
            continue
        diff[function] = {}
        for metric in curr_profile[function]:
            if metric == 'uid' or metric == 'image' or metric == 'symbol' or metric == 'inlined from':
                continue
            try:
                after_str = curr_profile[function][metric]
                after = float(after_str) if after_str else 0.0
                before_str = prev_profile[function][metric]
                before = float(before_str) if before_str else 0.0
            except ValueError:
                continue
            diff_value = after - before
            diff_value_str = f'{float(f"{diff_value:.2g}"):g}'
            diff_value_str = diff_value_str if (
                diff_value <= 0) else '+' + diff_value_str
            diff[function][metric] = [diff_value_str, '-']

            if diff_value == 0:
                diff[function][metric][1] = '0%'
            elif before == 0:
                diff[function][metric][1] = 'undefined'
            else:
                diffrel_value = diff_value / abs(before) * 100
                diffrel_value_str = f'{float(f"{diffrel_value:.2g}"):g}'
                diffrel_value_str = (diffrel_value_str if (
                    diffrel_value <= 0) else '+' + diffrel_value_str) + '%'
                diff[function][metric][1] = diffrel_value_str

    for function in prev_profile:
        if function not in curr_profile:
            removed_funcs.append(function)

    rank_diff = {function: 'unranked' for function in diff.keys()}
    for rank1, function1 in enumerate(prev_profile.keys(), 1):
        for rank2, function2 in enumerate(curr_profile.keys(), 1):
            if function1 == function2:
                rank_diff_value = rank2 - rank1
                rank_diff_value_str = str(abs(rank_diff_value))
                if rank_diff_value > 0:
                    rank_diff_value_str = '\u2191' + rank_diff_value_str
                if rank_diff_value < 0:
                    rank_diff_value_str = '\u2193' + rank_diff_value_str
                else:
                    rank_diff_value_str = '-'
                rank_diff[function1] = rank_diff_value_str

    for function in diff:
        diff[function]['Rank'] = rank_diff[function]

    profile_diff = {'new': new_funcs, 'removed': removed_funcs, 'diff': diff}

    return profile_diff


def diff_main(curr_csv_path, prev_csv_path, out_name):
    curr_profile = profile_csv_to_profile_dict(curr_csv_path)
    prev_profile = profile_csv_to_profile_dict(prev_csv_path)

    if curr_profile or prev_profile:
        profile_diff = diff_profiles(curr_profile, prev_profile)

        new_funcs = profile_diff['new']
        removed_funcs = profile_diff['removed']

        if new_funcs:
            print(f"New functions: {new_funcs} - not in {prev_csv_path}")
        if removed_funcs:
            print(
                f"Removed functions: {removed_funcs} - not in {curr_csv_path}")

        with open(out_name + '.json', 'w') as json_out:
            json.dump(profile_diff, json_out)

        diff_df = diff_dict_to_diff_df(profile_diff['diff'])
        diff_df.to_csv(out_name + '.csv')
        print(f"Diff report output to ./{out_name}.json and ./{out_name}.csv")


if __name__ == "__main__":
    if len(sys.argv) < 4:
        print(
            "Incorrect usage. Provide two csv file path arguments and an output file name.")
    else:
        diff_main(sys.argv[1], sys.argv[2], sys.argv[3])
