/* --------------------------------------------------------------------------
 * Copyright (c) 2013-2024 Arm Limited (or its affiliates). All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *      Name:    main.c
 *      Purpose: RTOS2 example program
 *
 *---------------------------------------------------------------------------*/

#include "RTE_Components.h"
#include CMSIS_device_header

#include <stdio.h>
#include "rtx_os.h"
#include "barman.h"

/* Provides access to the idle and timer thread objects */
extern const osRtxConfig_t osRtxConfig;

/* ids of threads created by this app */
osThreadId_t tid_phaseA;                /* Thread id of thread: phaseA      */
osThreadId_t tid_phaseB;                /* Thread id of thread: phaseB      */
osThreadId_t tid_phaseC;                /* Thread id of thread: phaseC      */
osThreadId_t tid_phaseD;                /* Thread id of thread: phaseD      */
osThreadId_t tid_clock;                 /* Thread id of thread: clock       */
osThreadId_t tid_app_main;              /* Thread id of thread: app_main    */

/* names of threads created by this app */
const osThreadAttr_t phaseA_attr   = { .name = "phaseA" };
const osThreadAttr_t phaseB_attr   = { .name = "phaseB" };
const osThreadAttr_t phaseC_attr   = { .name = "phaseC" };
const osThreadAttr_t phaseD_attr   = { .name = "phaseD" };
const osThreadAttr_t clock_attr    = { .name = "clock" };
const osThreadAttr_t app_main_attr = { .name = "app_main" };

struct phases_t {
  int_fast8_t phaseA;
  int_fast8_t phaseB;
  int_fast8_t phaseC;
  int_fast8_t phaseD;
} g_phases;


/*----------------------------------------------------------------------------
 *      Function 'signal_func' called from multiple threads
 *---------------------------------------------------------------------------*/
__attribute__((noinline)) void signal_func (osThreadId_t tid)  {
  unsigned int i;
  barman_annotate_marker(BM_ANNOTATE_COLOR_WHITE, "signal to clock #1");
  osThreadFlagsSet(tid_clock, 0x0100);      /* set signal to clock thread    */
  for( i=0; i <= 100000; i++) { __NOP(); }  /* some time-consuming work      */
  osDelay(500);                             /* delay 500ms                   */
  barman_annotate_marker(BM_ANNOTATE_COLOR_BLACK, "signal to clock #2");
  osThreadFlagsSet(tid_clock, 0x0100);      /* set signal to clock thread    */
  for( i=0; i <= 100000; i++) { __NOP(); }  /* some time-consuming work      */
  osDelay(500);                             /* delay 500ms                   */
  barman_annotate_marker(BM_ANNOTATE_COLOR_CYAN, "signal to thread");
  osThreadFlagsSet(tid, 0x0001);            /* set signal to thread 'thread' */
  for( i=0; i <= 100000; i++) { __NOP(); }  /* some time-consuming work      */
  osDelay(500);                             /* delay 500ms                   */
}

/*----------------------------------------------------------------------------
 *      Thread 1 'phaseA': Phase A output
 *---------------------------------------------------------------------------*/
void phaseA (void *argument) {
  unsigned int i;
  for (;;) {
    barman_annotate_channel(1, BM_ANNOTATE_COLOR_RED, "phaseA");
    osThreadFlagsWait(0x0001, osFlagsWaitAny, osWaitForever);   /* wait for an event flag 0x0001 */
    for( i=0; i <= 100000; i++) { __NOP(); }                    /* some time-consuming work      */
    g_phases.phaseA = 1;
    signal_func(tid_phaseB);                                    /* call common signal function   */
    g_phases.phaseA = 0;
    printf("This is phaseA\n");
    barman_annotate_channel(1, BM_ANNOTATE_COLOR_RED, NULL);
  }
}

/*----------------------------------------------------------------------------
 *      Thread 2 'phaseB': Phase B output
 *---------------------------------------------------------------------------*/
void phaseB (void *argument) {
  unsigned int i;
  for (;;) {
    barman_annotate_channel(2, BM_ANNOTATE_COLOR_GREEN, "phaseB");
    osThreadFlagsWait(0x0001, osFlagsWaitAny, osWaitForever);   /* wait for an event flag 0x0001 */
    for( i=0; i <= 100000; i++) { __NOP(); }                    /* some time-consuming work      */
    g_phases.phaseB = 1;
    signal_func(tid_phaseC);                                    /* call common signal function   */
    g_phases.phaseB = 0;
    printf("This is phaseB\n");
    barman_annotate_channel(2, BM_ANNOTATE_COLOR_GREEN, NULL);
  }
}

/*----------------------------------------------------------------------------
 *      Thread 3 'phaseC': Phase C output
 *---------------------------------------------------------------------------*/
void phaseC (void *argument) {
  unsigned int i;
  for (;;) {
    barman_annotate_channel(3, BM_ANNOTATE_COLOR_BLUE, "phaseC");
    osThreadFlagsWait(0x0001, osFlagsWaitAny, osWaitForever);   /* wait for an event flag 0x0001 */
    for( i=0; i <= 100000; i++) { __NOP(); }                    /* some time-consuming work      */
    g_phases.phaseC = 1;
    signal_func(tid_phaseD);                                    /* call common signal function   */
    g_phases.phaseC = 0;
    printf("This is phaseC\n");
    barman_annotate_channel(3, BM_ANNOTATE_COLOR_BLUE, NULL);
  }
}

/*----------------------------------------------------------------------------
 *      Thread 4 'phaseD': Phase D output
 *---------------------------------------------------------------------------*/
void phaseD (void *argument) {
  unsigned int i;
  for (;;) {
    barman_annotate_channel(4, BM_ANNOTATE_COLOR_YELLOW, "phaseD");
    osThreadFlagsWait(0x0001, osFlagsWaitAny, osWaitForever);   /* wait for an event flag 0x0001 */
    for( i=0; i <= 100000; i++) { __NOP(); }                    /* some time-consuming work      */
    g_phases.phaseD = 1;
    signal_func(tid_phaseA);                                    /* call common signal function   */
    g_phases.phaseD = 0;
    printf("This is phaseD\n");
    barman_annotate_channel(4, BM_ANNOTATE_COLOR_YELLOW, NULL);
  }
}

/*----------------------------------------------------------------------------
 *      Thread 5 'clock': Signal Clock
 *---------------------------------------------------------------------------*/
void clock (void *argument) {
  for (;;) {
    osThreadFlagsWait(0x0100, osFlagsWaitAny, osWaitForever);   /* wait for an event flag 0x0100 */
    barman_annotate_marker(BM_ANNOTATE_COLOR_GREEN, "clock delay start");
    osDelay(80);                                                /* delay  80ms                   */
    barman_annotate_marker(BM_ANNOTATE_COLOR_RED, "clock delay end");
  }
}

/*----------------------------------------------------------------------------
 *      Main: Initialize and start the application
 *---------------------------------------------------------------------------*/
void app_main (void *argument) {
  printf("CMSIS RTX5 RTOS Example for AC6\n");
  printf("===============================\n");
  printf("Starting some threads...\n");

  osThreadFlagsSet(tid_phaseA, 0x0001); /* set signal to phaseA thread   */

  osDelay(osWaitForever);
  while(1);
}

/*----------------------------------------------------------------------------
 *      Main: Initialize and start the RTOS2 Kernel
 *---------------------------------------------------------------------------*/

extern void uart_init(void);     // in uart.c
static void enable_barman(void); // later in this file

int main (void) {
  uart_init();      // in uart.c

  // System Initialization
  SystemCoreClockUpdate();

  osKernelInitialize();                 // Initialize CMSIS-RTOS

  tid_phaseA   = osThreadNew(phaseA,   NULL, &phaseA_attr);
  tid_phaseB   = osThreadNew(phaseB,   NULL, &phaseB_attr);
  tid_phaseC   = osThreadNew(phaseC,   NULL, &phaseC_attr);
  tid_phaseD   = osThreadNew(phaseD,   NULL, &phaseD_attr);
  tid_clock    = osThreadNew(clock,    NULL, &clock_attr);
  tid_app_main = osThreadNew(app_main, NULL, &app_main_attr);    // Create application main thread

  enable_barman();                      // Enable barman
  if (osKernelGetState() == osKernelReady) {
    osKernelStart();                    // Start thread execution
  }

  while(1);
}


// ==== barman support code ====

#include "../Source/rtx_lib.h" // provides osRtxThreadxxx functions and os_thread_t type

/*
 * Perform the necessary initialization of the bare-metal agent
 */
static void enable_barman(void)
{
   /* For M-class, the cycle counter provides the timestamps, so convert it to nS by multiplying by 10**9 and dividing by the clock frequency in Hz */
    const struct bm_protocol_clock_info clock_info = { .timestamp_base = 0,
                                                       .timestamp_multiplier = 1000000000,
                                                       .timestamp_divisor = 25000000, /* 25MHz system freq of AN521 FPGA */
                                                       .unix_base_ns = 0 };

#if BM_CONFIG_MAX_TASK_INFOS > 0
    const struct bm_protocol_task_info task_entries[] =
    {
        { (bm_task_id_t)tid_phaseA,   "phaseA" },
        { (bm_task_id_t)tid_phaseB,   "phaseB" },
        { (bm_task_id_t)tid_phaseC,   "phaseC" },
        { (bm_task_id_t)tid_phaseD,   "phaseD" },
        { (bm_task_id_t)tid_clock,    "clock" },
        { (bm_task_id_t)tid_app_main, "app_main" },
        { (bm_task_id_t)(osRtxConfig.idle_thread_attr->cb_mem), "osRtxIdleThread" },
        { (bm_task_id_t)(osRtxConfig.timer_thread_attr->cb_mem), "osRtxTimerThread" },
    };
#endif

    /* Initialize barman but if there is a problem we will loop here */
    while (!barman_initialize_with_itm_interface("RTX5 Cortex-M33 Streamline bare-metal example", &clock_info,
#if BM_CONFIG_MAX_TASK_INFOS > 0
                              /* All the tasks */
                              8, task_entries,
#endif
#if BM_CONFIG_MAX_MMAP_LAYOUTS > 0
                              /* We only have one image for all tasks so we don't need to provide these */
                              0, BM_NULL,
#endif
                              1));

    /* Now we are ready to enable sampling */
    barman_enable_sampling();
}

/* Allow barman to read the current thread id from RTX */
bm_task_id_t barman_ext_get_current_task_id(void)
{
  return (bm_task_id_t)osRtxThreadGetRunning();
}

/* Allow RTX to inform barman when a task switch occurs */
extern void $Super$$osRtxThreadSwitch(os_thread_t *thread);

void $Sub$$osRtxThreadSwitch(os_thread_t *thread)
{
  $Super$$osRtxThreadSwitch(thread);                               // Call the original osRtxThreadSwitch
  barman_record_task_switch(BM_TASK_SWITCH_REASON_PREEMPTED);      // Record the task switch
}
