/**
 * Copyright (C) 2016-2023 by Arm Limited (or its affiliates). All rights reserved.
 * Use, modification and redistribution of this file is subject to your possession of a
 * valid End User License Agreement for the Arm Product of which these examples are part of
 * and your compliance with all applicable terms and conditions of such license agreement.
 */
#include <stdio.h>
#include <stdarg.h>

#include "barman.h"
#include "device.h"
#include "target.h"
#include "generic_timer.h"
#include "gic.h"
#include "v8_aarch64.h"

/* A 2MB buffer for barman to store the collected data */
bm_uint8 buffer_for_barman[0x200000];

bm_int32 timer_reload_value;;

/*
 * Get the timestamp for barman.
 *
 * We are going to use the system counter.
 */
bm_uint64 barman_ext_get_timestamp()
{
    return getSystemCounter();
}

/*
 * If we decide to enable logging within barman,
 * just print the messages using semihosting.
 */
#if BM_CONFIG_ENABLE_DEBUG_LOGGING
void barman_ext_log_debug(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}
#endif

#if BM_CONFIG_ENABLE_LOGGING
void barman_ext_log_error(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}

void barman_ext_log_warning(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}

void barman_ext_log_info(const char * message, ...)
{
    va_list argp;
    va_start(argp, message);
    vprintf(message, argp);
    va_end(argp);
}
#endif

#if BM_CONFIG_MAX_CORES > 1
/*
 * Barman needs to know how to map MPIDR values
 * to a unique core number in a contiguous range starting at zero
 */
bm_uint32 barman_ext_map_multiprocessor_affinity_to_core_no(bm_uintptr mpidr)
{
    return mpidr_to_core_number(mpidr);
}

/*
 * Barman needs to know how to map MPIDR values
 * to a cluster number in a contiguous range starting at zero
 */
bm_uint32 barman_ext_map_multiprocessor_affinity_to_cluster_no(bm_uintptr mpidr)
{
    return mpidr_to_cluster_number(mpidr);
}
#endif

/*
 * If we want to provide Streamline with task information
 * we must provide barman with a way to get the current task ID
 */
#if BM_CONFIG_MAX_TASK_INFOS > 0
bm_task_id_t barman_ext_get_current_task_id(void)
{
    /* In this example we have one task running on each core */
    return GetCoreNumber();
}
#endif

/*
 * Initialize barman and start sampling.
 */
void initBarman(void)
{
    const bm_uint64 system_counter_frequency = getSystemCounterFrequency();

    const struct bm_protocol_clock_info clock_info = {
        .timestamp_base = 0,
        .timestamp_multiplier = 1000000000,
        .timestamp_divisor = system_counter_frequency,
        .unix_base_ns = 0
    };

#if BM_CONFIG_MAX_TASK_INFOS > 0
    const struct bm_protocol_task_info task_entries[8] =
    {
        {0, "core 0"},
        {1, "core 1"},
        {2, "core 2"},
        {3, "core 3"},
        {4, "core 4"},
        {5, "core 5"},
        {6, "core 6"},
        {7, "core 7"}
    };
#endif

    /*
     * barman_initialize has a different prototype
     * depending on what options we have chosen
     */
    const bm_bool init_success =

#if (BM_CONFIG_USE_DATASTORE == BM_CONFIG_USE_DATASTORE_LINEAR_RAM_BUFFER) || (BM_CONFIG_USE_DATASTORE == BM_CONFIG_USE_DATASTORE_CIRCULAR_RAM_BUFFER)

    /* A RAM buffer datastore */
    barman_initialize(buffer_for_barman, sizeof(buffer_for_barman),

#elif (BM_CONFIG_USE_DATASTORE == BM_CONFIG_USE_DATASTORE_STM) && defined(STM_CONFIGURATION_REGISTERS) && defined(STM_EXTENDED_STIMULUS_PORTS)

    /* The STM datastore */
    barman_initialize_with_stm_interface((void *) STM_CONFIGURATION_REGISTERS, (void *) STM_EXTENDED_STIMULUS_PORTS,

#else
#error "Unsupported datastore backend"
#endif

    "barman AArch64 example", &clock_info,
#if BM_CONFIG_MAX_TASK_INFOS > 0
    8, task_entries,
#endif
#if BM_CONFIG_MAX_MMAP_LAYOUTS > 0
    /* We only have one image for all tasks so we don't need to provide these */
    0, BM_NULL,
#endif
    1);

    /* If there is a problem initializing barman we will loop here */
    while(!init_success);

    /* Now we are ready to enable sampling */
    barman_enable_sampling();

    /* We will sample at 10kHz */
    timer_reload_value = system_counter_frequency / 10000;
}

/*
 * Interrupt handler that calls barman's sample function,
 * passing in the return address of the interrupt as the PC value.
 */
void barmanInterruptHandler(void)
{
    const unsigned int core = GetCoreNumber();
    /* We are making a color unique to each core */
    const unsigned int color = BM_ANNOTATE_COLOR_RGB(32 * core, 32 * (core + 4), 32 * (8 - core));
    char buffer[100];

    const void* pc;
    asm ("mrs %0, ELR_EL1" : "=r"(pc));
    barman_sample_counters_with_program_counter(pc);

    snprintf(buffer, 100, "Sample on core %u", core);
    barman_annotate_marker(color, buffer);

    snprintf(buffer, 100, "Hi %u", core);
    barman_annotate_channel(core, color, buffer);

    setEl1PhysicalTimerTimerValue(timer_reload_value);
}

void initBarmanInterrupt(void)
{
    const unsigned int core = GetCoreNumber();
    const unsigned int cluster = GetClusterNumber();
    char buffer[100];

    snprintf(buffer, 100, "Cluster %u group", cluster);
    // This might get called more than once for the cluster but it doesn't matter
    barman_annotate_name_group(cluster, buffer);

    snprintf(buffer, 100, "Core %u channel", core);
    barman_annotate_name_channel(core, cluster, buffer);

    // Set up the timer
    setEl1PhysicalTimerTimerValue(timer_reload_value);
    setEl1PhysicalTimerEnabled(true);
    setEl1PhysicalTimerInterruptEnabled(true);

    // Set up the GIC
    // 30 is the INTID of the EL1 physical timer
    SetIRQPriority(30, 0xb0);  // Set INTID 30 to priority to 0xb0
    EnableIRQ(30);             // Enable INTID 30

}


