/* Copyright (C) 2021-2023 by Arm Limited. All rights reserved. */

#include "android/Utils.h"

#include "GatorCLIParser.h"
#include "Logging.h"
#include "lib/FsEntry.h"
#include "lib/LineReader.h"
#include "lib/Popen.h"
#include "lib/Process.h"

#include <algorithm>
#include <array>
#include <optional>
#include <string>
#include <string_view>
#include <utility>
#include <vector>

namespace android_utils {

    constexpr std::string_view ARG_SHORT_OPTION_START = "-";
    constexpr std::string_view WHITE_SPACE = " ";

    bool isArgMatched(const std::string & arg, const struct option & option)
    {
        return (arg == option.name) || (arg.length() == 1 && arg[0] == (char) option.val);
    }

    std::optional<std::vector<std::string>> getGatorArgsWithAndroidOptionsReplaced(
        const std::vector<std::pair<std::string, std::optional<std::string>>> & gatorArgValuePairs)
    {
        std::vector<std::string> result;
        bool hasPackageName = false;
        bool hasWaitProcessOrApp = false;
        bool hasPackageArg = false;

        for (auto const & argValuePair : gatorArgValuePairs) {
            auto arg = argValuePair.first;
            auto val = argValuePair.second;

            if (isArgMatched(arg, WAIT_PROCESS) || isArgMatched(arg, APP)) {
                hasWaitProcessOrApp = true;
                break;
            }
            if (isArgMatched(arg, ANDROID_PACKAGE)) {
                hasPackageArg = true;
                if (!val.has_value() || val.value().empty()
                    || val.value().find_first_not_of(WHITE_SPACE.data()) == std::string::npos) {
                    break;
                }
                hasPackageName = true;
                result.push_back(ARG_SHORT_OPTION_START.data() + std::string(1, (char) WAIT_PROCESS.val));
                result.push_back(val.value());
            }
            else if (!isArgMatched(arg, ANDROID_ACTIVITY)) {
                result.push_back(ARG_SHORT_OPTION_START.data() + arg);
                if (val) {
                    result.push_back(val.value());
                }
            }
        }
        if (hasWaitProcessOrApp || !hasPackageName || !hasPackageArg) {
            return std::nullopt;
        }
        return {result};
    }

    bool copyApcToActualPath(const std::string & androidPackageName, const std::string & apcPathInCmdLine)
    {
        auto canCreateApc = canCreateApcDirectory(apcPathInCmdLine);
        if (!canCreateApc) {
            return false;
        }
        auto origApcDir = lib::FsEntry::create(canCreateApc.value());

        const auto targetTarFile = origApcDir.path() + ".tar";
        auto targetTarFileFsEnrty = lib::FsEntry::create(targetTarFile);

        //create tar copy apc to cmdline output path
        int copyResult =
            gator::process::runCommandAndRedirectOutput("run-as " + androidPackageName + " tar -c " + origApcDir.name(),
                                                        std::make_optional(targetTarFile));
        if (copyResult != 0) {
            LOG_ERROR("Zip tar file '/data/data/%s/%s' to '%s' failed ",
                      androidPackageName.c_str(),
                      origApcDir.name().c_str(),
                      targetTarFile.c_str());
            return false;
        }
        auto destination = origApcDir.parent() ? origApcDir.parent().value().path() : "/data/local/tmp";
        //unzip tar
        auto unzipResult = gator::process::runCommandAndRedirectOutput("tar --no-same-owner -xf " + targetTarFile
                                                                           + " -C " + destination,
                                                                       std::nullopt);
        if (unzipResult != 0) {
            LOG_WARNING("Unzipping tar file '%s' to '%s' failed ", targetTarFile.c_str(), destination.c_str());
        }
        gator::process::runCommandAndRedirectOutput("run-as " + androidPackageName + " rm -r " + origApcDir.name(),
                                                    std::nullopt);
        if (!targetTarFileFsEnrty.remove()) {
            LOG_WARNING("Failed to removed tar file '%s'", targetTarFileFsEnrty.path().c_str());
        }
        return unzipResult == 0;
    }

    std::optional<std::string> getApcFolderInAndroidPackage(const std::string & appCwd,
                                                            const std::string & targetApcPath)
    {
        const auto origApcDir = lib::FsEntry::create(targetApcPath);
        if (origApcDir.name().empty()) {
            return std::nullopt;
        }
        return appCwd + "/" + origApcDir.name();
    }

    std::optional<std::string> canCreateApcDirectory(const std::string & targetApcPath)
    {
        auto apcDir = lib::FsEntry::create(targetApcPath);
        std::string apcPathWithEtn(apcDir.path());
        //check if path ends with .apc, if not append .apc to folder
        if (apcPathWithEtn.size() > 0
            && apcPathWithEtn.compare(apcPathWithEtn.size() - 4, apcPathWithEtn.size(), ".apc") != 0) {
            apcPathWithEtn += ".apc";
        }
        auto origApcDir = lib::FsEntry::create(apcPathWithEtn);
        if (origApcDir.exists()) {
            origApcDir.remove_all();
            if (origApcDir.exists()) {
                LOG_ERROR("Desitination folder exists '%s' and could not be deleted.", apcPathWithEtn.c_str());
                return {};
            }
        }
        if (!origApcDir.parent() || !origApcDir.parent().value().exists()) {
            if (!origApcDir.create_directory()) {
                LOG_ERROR("Failed to create a destination folder '%s'.", apcPathWithEtn.c_str());
                return {};
            };
        }
        return {origApcDir.path()};
    }

    // Trim any newline characters off the right end of the string
    static void rightTrim(std::string & s)
    {
        s.erase(std::remove(s.begin(), s.end(), '\n'), s.cend());
    }

    static bool findAndroidPackage(const lib::PopenResult invokedCmd, const std::string & package)
    {
        std::string currentPackage;
        lib::LineReader reader(invokedCmd.out);
        lib::LineReaderResult moreLines {};
        while (moreLines.ok()) {
            currentPackage.clear();
            moreLines = reader.readLine(currentPackage);
            rightTrim(currentPackage);
            if (currentPackage == "package:" + package) {
                LOG_DEBUG("Package found: " + package);
                return true;
            }
        }
        return false;
    }

    bool packageExists(const std::string & desiredPackage)
    {
        constexpr std::string_view PM = "pm";
        constexpr std::string_view LIST = "list";
        constexpr std::string_view PKGS = "packages";

        const std::array<const char *, 4> cmd = {PM.data(), LIST.data(), PKGS.data(), nullptr};

        const lib::PopenResult pkgListingInvocation = lib::popen(cmd);
        bool packageFound = false;
        if (pkgListingInvocation.pid >= 0) {
            packageFound = findAndroidPackage(pkgListingInvocation, desiredPackage);

            // positive number or zero is the exit status from the command
            // negative number indicates a problem waiting, so we don't know whether
            // the command was ok
            const int pclose_res = lib::pclose(pkgListingInvocation);
            if (pclose_res == 0) {
                return packageFound;
            }

            LOG_ERROR("Could not discover installed packages, pclose of 'pm list packages' reported code %d",
                      pclose_res);
            return false;
        }

        LOG_WARNING("Could not discover installed packages, is the 'pm' command installed?");
        return false;
    }

}
