// ------------------------------------------------------------
// ARMv8 AArch64 C library Startup Code
//
//
// Copyright (C) 2016-2023 by Arm Limited (or its affiliates). All rights reserved.
// Use, modification and redistribution of this file is subject to your possession of a
// valid End User License Agreement for the Arm Product of which these examples are part of 
// and your compliance with all applicable terms and conditions of such license agreement.
//
// ------------------------------------------------------------

#include "v8_system.h"

// AArch64 ARM C library startup add-in:

// To invalidate the AArch64 instruction cache after scatter-loading and before initialization of the stack and heap,
// it is necessary for the user to:
//
// * Implement instruction cache invalidation code in _platform_pre_stackheap_init.
// * Ensure all code on the path from the program entry up to and including _platform_pre_stackheap_init is located in a root region.
//
// In this example, this function is only called once, by the primary core

// For newlib we have written our own C library startup which calls _platform_pre_stackheap_init

#ifndef __ARMCC_VERSION /* newlib */
    .global __main
    .type __main, "function"
__main:
    bl  _platform_pre_stackheap_init

    // Zero the bss
    ldr x0, =__bss_start__ // Start of block
    mov x1, #0             // Fill value
    ldr x2, =__bss_end__   // End of block
    sub x2, x2, x0         // Length of block
    bl  memset

    // Set up the standard file handles
    bl  initialise_monitor_handles

    // Set up _fini and fini_array to be called at exit
    ldr x0, =__libc_fini_array
    bl  atexit

    // Call preinit_array, _init and init_array
    bl  __libc_init_array

    // Set argc = 1, argv[0] = "" and then call main
    .pushsection .data
    .align 3
argv:
    .dword arg0
    .dword 0
arg0:
    .byte 0
    .popsection

    mov x0, #1
    ldr x1, =argv
    bl main

    b exit // Will not return
#endif


    //
    // The ARM Architecture Reference Manual for ARMv8-A states:
    //
    //     Instruction accesses to Non-cacheable Normal memory can be held in instruction caches.
    //     Correspondingly, the sequence for ensuring that modifications to instructions are available
    //     for execution must include invalidation of the modified locations from the instruction cache,
    //     even if the instructions are held in Normal Non-cacheable memory.
    //     This includes cases where the instruction cache is disabled.
    //

    .global _platform_pre_stackheap_init
    .type _platform_pre_stackheap_init, "function"
_platform_pre_stackheap_init:
    dsb ish     // ensure all previous stores have completed before invalidating
    ic  ialluis // I cache invalidate all inner shareable to PoU (which includes secondary cores)
    dsb ish     // ensure completion on inner shareable domain   (which includes secondary cores)
    isb

    // Scatter-loading is complete, so enable the caches here, so that the C-library's mutex initialization later will work
    mrs x1, SCTLR_EL1
    orr x1, x1, #SCTLR_ELx_C
    orr x1, x1, #SCTLR_ELx_I
    msr SCTLR_EL1, x1
    isb

    ret
