/*
 * ARMv8 - Main
 *
 *
 * Copyright (C) 2011-2023 by Arm Limited (or its affiliates). All rights reserved.
 * Use, modification and redistribution of this file is subject to your possession of a
 * valid End User License Agreement for the Arm Product of which these examples are part of
 * and your compliance with all applicable terms and conditions of such license agreement.
 */

#include <stdlib.h>
#include <stdio.h>

#include "gic.h"
#include "MP_Mutexes.h"
#include "primes.h"
#include "v8_aarch64.h"
#include "barman_interrupt.h"

// compile-time control for the number of CPUs in the device
#define nCPUs 8

static unsigned int cpu_finished_count = 0;

/*
 * void MainApp(void)
 *
 *    called by all CPUs to run SMP applications, this is the
 *    application start point for all secondary CPUs, and is called
 *    after other startup code has executed on the primary CPU
 *
 * Inputs
 *   <none>
 *
 * Returns
 *   <function does not return>
 */
__attribute__((noreturn)) void MainApp(void)
{
    const unsigned long core = GetCoreNumber();

    if (core >= nCPUs)
    {
      _mutex_acquire(&print_lock);
      printf("Error: More CPU cores are running than the maximum this program expects\n");
      printf("Change the value of nCPUs.  Exiting...\n");
      _mutex_release(&print_lock);
      exit(1);
    }

    initBarmanInterrupt();

    // Enable interrupts
    asm ("msr DAIFClr, #0xF");
    EnableGICC();

    calculatePrimes(core);

    _mutex_acquire(&print_lock);
    printf("CPU %lu: finished\n", core);
    _mutex_release(&print_lock);

    if (__atomic_add_fetch(&cpu_finished_count, 1, __ATOMIC_RELAXED) < nCPUs)
    {
      /*
       * CPUs that finish early wait for termination
       */
      while(1)
        asm("wfi");
    }
    else
    {
      /*
       * The last CPU to finish terminates the program
       */
      exit(0);
    }
}

/*
 * int main(void)
 *    the application start point for the primary CPU - bring up the
 *    secondary CPUs and then call MainApp
 *
 *  Inputs
 *    <none>
 *
 *  Returns
 *    subroutine does not return
 */
int main(void)
{
    printf("\r\nArmv8 Prime Number Generator Example\r\n\r\n");

    initBarman();

    initPrimes(); // Initialize the primes just once, including print_lock

    asm("dsb OSHST"); // Ensure the side effects of initPrimes are seen before waking the CPUs

    /*
     * send SGI15 to the other (i.e. secondary) CPUs to
     * wake them up
     *
     * these should be group1 ("non-secure") interrupts
     */
    SendSGI(15, 1 /* all except us */, 0 /* ignored target */, 1);

    /*
     * start the main application with the other CPUs
     *
     * note: MainApp() does not return
     */
    MainApp();
}
