##
## For each frame, print a list of OpenGL ES draw calls with blending enabled.
##
## Two implementations are presented, a slow linear implementation and a faster
## threaded implementation that takes advantage of Jython threading functionality.
##
## Jython can natively use the Java threading model and isn't blocked by a
## Global Interpreter Lock.
##
## To see both implementations, execute this script inside the Scripting View.
## This will add linear() and threaded() to the global scope.
## You can then execute them directly in the command line and compare them.
##
## The difference is most obvious in large trace files with many function calls.
## For small traces, the overhead of setting up threads may cause threaded()
## to take longer than linear().
##

from threading import Thread
from timeit import default_timer as timer

# Collect every OpenGL ES draw call with blending enabled.
draw_calls = [ fc for fc in trace.processes()[0].function_calls() if (fc.is_draw_call()) and fc.states()['GL_BLEND'] ]

def functions_in_frame(functions, frame):
    """Returns a sublist of functions that are attributed to the provided frame."""

    # Note: There are more efficient ways to do this (e.g. iterate over frame.function_calls())
    # This is more syntactically concise and is deliberately inefficient to demonstrate threading performance.

    # Find function calls in the provided list that are also in the frame's function calls
    return [ fc for fc in functions if fc in frame.function_calls() ]

class FrameAnalysisThread(Thread):
    """Thread subclass that searches for function calls with blending enabled in the frame provided in the constructor."""

    def __init__(self, frame, group=None, target=None, name=None, args=(), kwargs={}):

        # Override the constructor to store a reference to the frame object.
        self.frame = frame
        super(FrameAnalysisThread, self).__init__(group, target, name, args, kwargs)

    def run(self):
        # Print draw calls with blending enabled
        print("{0}: {1}\n".format(self.frame, functions_in_frame(draw_calls, self.frame)))

def linear():
    """Iteratively runs through each frame and prints OpenGL ES draw calls with blending enabled. Runs slowly but frames are printed in order."""

    print("Searching for OpenGL ES draw calls in each frame...\n")
    start = timer()

    for frame in trace.processes()[0].frames():

        # Print draw calls with blending enabled
        print("{0}: {1}\n".format(frame, functions_in_frame(draw_calls, frame)))

    print("Time taken: {} seconds".format(timer() - start))

def threaded():
    """Creates and launches a thread for each frame. Each thread will print OpenGL ES draw calls with blending enabled for that frame.
    Runs quickly but frame order is not preserved."""

    print("Searching for OpenGL ES draw calls in each frame...\n")
    start = timer()

    threads = [ FrameAnalysisThread(frame=frame) for frame in trace.processes()[0].frames() ]

    # Start each thread
    for thread in threads:
        thread.start()

    # Wait for threads to finish
    for thread in threads:
        thread.join()

    print("Time taken: {} seconds".format(timer() - start))

print "Functions threaded() and linear() have been added to the global scope. Execute them directly from the command line and compare the results."