##
## Attach a note to each draw call that has blending enabled
##
## This script demonstrates how to:
##  - Add notes to function calls which have interesting properties
##  - Access the global state of an OpenGL ES application
##  - Look up an asset by the handle used to reference it in a function call
##  - Access the state of a Vulkan command buffer at a specific command
##


# Iterate through all the processes in the trace.
# The "trace" variable is automatically exported in the scripting console when a trace is opened.
for process in trace.processes():
    # Get the list of all the function calls in the process
    function_calls = process.function_calls()

    monitor.set_task(str(process), len(process.egl_and_gles_function_calls()))

    # In OpenGL ES, blending is a global state
    for function_call in process.egl_and_gles_function_calls():
        if function_call.is_draw_call() and function_call.states().get("GL_BLEND"):
            function_call.set_note("Blending is enabled")
        monitor.increment_progress(1)

    monitor.set_task(str(process), len(process.vulkan_function_calls()))

    # In Vulkan, blending is part of the graphics pipeline
    for function_call in process.vulkan_function_calls():
        if function_call.is_draw_command():
            # Get the handle of the command buffer the draw call was enqueued in
            command_buffer_handle = function_call.parameters()["commandBuffer"]

            # Look up the command buffer asset
            command_buffer = function_call.assets()["Command Buffers"][int(command_buffer_handle, 16)]

            # Get the most recently added command to the command buffer, which will be the one enqueued by function_call
            this_command = command_buffer["Commands"].values()[-1]

            # Get the blend state of the command buffer at this_command
            try:
                blend_state = this_command["Command Buffer State"]["Graphics Pipeline"]["colorBlendState"]

            except (KeyError, TypeError):
                # The command doesn't have the required state set.
                blend_state = NONE;

            # If the blend state is non-null at this command, iterate through each attachment in the bound graphics pipeline
            if blend_state:
                attachment_count = 0
                blend_enabled_count = 0
                if blend_state["pAttachments"]:
                    for attachment in blend_state["pAttachments"].values():
                        attachment_count += 1
                        # The value of blendEnable will be either "VK_FALSE" or "VK_TRUE", which implicitly resolve to False and True, respectively
                        if attachment["blendEnable"]:
                            blend_enabled_count += 1

                    if blend_enabled_count:
                        function_call.set_note("{0} of {1} color attachment(s) have blending enabled".format(blend_enabled_count, attachment_count))
        monitor.increment_progress(1)

    monitor.finish()