##
## For each process, retrieve the penultimate frame.
## Retrieve the assets from the final function call in the frame.
## Look for compressed textures and track how many have been found.
## Report the percentage of textures that were compressed.
##
## This script demonstrates how to extract asset information from a
## function call's state record.
##

def is_texture_initialized(texture_asset):
    """Returns True if the texture is initialized (i.e. has a target), False otherwise."""
    target_texture = texture_asset.get('Target').__repr__()
    return "None" not in target_texture

def is_texture_compressed(texture_asset):
    """Inspects the properties of a texture asset and returns true if the texture contains
    a compressed internal format (otherwise returns false)."""

    # The target property of the texture asset tells us what type of texture we are dealing
    # with e.g. 2D, 3D, cube-mapped
    target_texture = texture_asset.get('Target').__repr__()

    # Switch based on the type of texture

    # Covers GL_TEXTURE_2D_ARRAY, GL_TEXTURE_2D_MULTISAMPLE_ARRAY, GL_TEXTURE_CUBE_MAP_ARRAY
    if "ARRAY" in target_texture:

        layer = texture_asset.get("Layers").values()[0]

        # Check value exists
        if layer.__repr__() != "'None'":

            if "2D" in target_texture:

                # Layers are array equivalent of mipmaps and have internal format info
                level_zero = layer.values()[0]

                if level_zero.__repr__() != "'None'":

                    if "COMPRESSED" in level_zero['Internal Format'].__repr__():

                        return True

            else:
                # Cube map

                # Cube map layers contain the array equivalent of faces
                first_face = layer.values()[0]

                if first_face.__repr__() != "'None'":

                    # Extract the first face format
                    first_face_format = first_face.values()[0]['Internal Format']

                    if "COMPRESSED" in first_face_format.__repr__():

                        return True



    # Covers GL_TEXTURE_2D, GL_TEXTURE_2D_MULTISAMPLE, GL_TEXTURE_3D, GL_TEXTURE_BUFFER
    elif ("2D" in target_texture) or ("3D" in target_texture) or ("BUFFER" in target_texture):

        # Contains info about the texture
        texture = texture_asset.get("Texture")

        if texture.__repr__() != "'None'":

            # This value represents the first mipmap level
            level_zero = texture.values()[0]

            if level_zero.__repr__() != "'None'":

                # Check the internal format for compression
                if "COMPRESSED" in level_zero['Internal Format'].__repr__():

                    return True

    # Covers GL_TEXTURE_CUBE_MAP
    elif "CUBE" in target_texture:

        # Cube mapped textures have faces rather than mipmaps/layers
        first_face = value.get('Cube Map Faces').values()[0]

        if first_face.__repr__() != "'None'":

            # Extract the first face
            first_face_format = first_face.values()[0]['Internal Format']

            if "COMPRESSED" in first_face_format.__repr__():

                return True

    else:

        raise ValueError("Unexpected OpenGL ES texture target: {}".format(target_texture))

    # No compression found
    return False

# Iterate through all the processes in the trace.
# The "trace" variable is automatically exported in the scripting console when a trace is opened.
for process in trace.processes():

    compressed = 0
    uncompressed = 0

    print("{}:".format(process))

    frames = process.frames()

    # Ask the user which frame they want to analyse.
    frame_number = raw_input("Which frame number do you want to analyse in " + str(process) + "?")

    # Make sure they've given a number and it's in range.
    if not frame_number.isdigit() or int(frame_number) >= len(frames):
        print "\tInvalid frame number (" + frame_number + ") chosen"
        continue

    print "\tAnalysing frame " + frame_number

    frame = frames[int(frame_number)]

    # Use the last function call in that frame.
    functions = frame.function_calls()
    if not functions:
        print "\tNo functions in the chosen frame"
        continue

    final_function = functions[-1]

    # Function calls contain a record of the state of the assets at the moment they were called.
    try:
        textures = final_function.assets()['Textures']

    except KeyError:
        print("\tError: Couldn't retrieve Textures from process {}. Please note that OpenCL and Vulkan are not supported by this script.".format(process))
        continue

    # The textures assets are exposed as a dictionary, but here we are only interested in the values,
    # so we can disregard the keys (which are just integers). We also want to ignore texture 0, which
    # is a special case in OpenGL ES, so we can slice the list from index 1 to the end.

    for value in textures.values()[1:]:

        if not is_texture_initialized(value):
            continue

        if is_texture_compressed(value):
            compressed += 1
        else:
            uncompressed += 1

    total = compressed + uncompressed

    # We have to write some logic here to avoid division by zero.
    if (total) == 0:
        print("\tNo textures found")

    else:
        percent = (compressed / float(compressed + uncompressed)) * 100
        print("\t{0:.2f}% of {1} textures are compressed".format(percent, total))