##
## Calculate the following statistics about each process in a Graphics Analyzer trace:
##  - The function calls with the longest duration
##  - The API calls with the longest cumulative duration
##  - The estimated percentage of time the application spent inside the driver
##
## This script teaches the basics of programmatically traversing a Graphics Analyzer trace.
##

def duration_micros(function_call):
    """Helper function to get the duration in microseconds that a function call was executing for."""
    return function_call.stop_time() - function_call.start_time()

# Iterate through all the processes in the trace.
# The "trace" variable is automatically exported in the scripting console when a trace is opened.
for process in trace.processes():

    print(str(process) + ":")
    # Get the list of all the function calls in the process
    function_calls = process.function_calls()

    # Early out if the process has no function calls
    if len(function_calls) == 0:
        print "\tNo function calls"
        continue

    ##
    ## Calculate and print the top 10 function calls with the longest duration.
    ##

    # Create a new list of function calls sorted by their CPU time duration, in descending order
    duration_sorted_function_calls = sorted(function_calls, key = lambda fc: duration_micros(fc), reverse=True)

    # Show the duration of up to the top 10 function calls
    top_function_call_count = min(10, len(duration_sorted_function_calls))

    print "\tThe {0} function calls that took the longest CPU time are:".format(top_function_call_count)

    for i in range(0, top_function_call_count):
        fc = duration_sorted_function_calls[i]
        print "\t\t{0} ({1}): {2} us".format(fc.name(), fc.index(), duration_micros(fc))

    ##
    ## Calculate and print the top 10 API calls with cumulatively the longest duration.
    ##

    # Accumulate the time spent in each API call.
    cumulative_times_micros = {}
    for fc in function_calls:
        name = fc.name()
        duration = duration_micros(fc)
        if name in cumulative_times_micros:
            cumulative_times_micros[name] += duration
        else:
            cumulative_times_micros[name] = duration

    # Sort API calls by the cumulative time spent in them, in descending order
    duration_sorted_api_calls = sorted(cumulative_times_micros, key = cumulative_times_micros.get, reverse=True)

    # Show the duration of up to the top 10 API calls.
    top_api_call_count = min(10, len(duration_sorted_api_calls))

    print "\tThe {0} API calls that cumulatively took the longest CPU time are:".format(top_api_call_count)

    for i in range(0, top_function_call_count):
        api_call_name = duration_sorted_api_calls[i]
        print "\t\t{0}: {1} us".format(api_call_name, cumulative_times_micros[api_call_name])

    ##
    ## Calculate and print the estimated percentage of time the process spent inside the driver.
    ##

    # Accumulate the time spent in all API calls.
    total_time_spent_in_api_calls_micros = sum(cumulative_times_micros.values())

    # Determine the total time spent in the application by subtracting the start time of the first function call from the stop time of the last.
    # This doesn't take into account application startup time, multiple threads, time the application was paused, overhead from frame capture or data transfer, etc.
    total_time_spent_in_application_micros = function_calls[len(function_calls) - 1].stop_time() - function_calls[0].start_time()

    percent_time_in_driver = (total_time_spent_in_api_calls_micros / float(total_time_spent_in_application_micros)) * 100;

    print "\t{0:.2f}% of CPU time in {1} was spent inside the driver.\n".format(percent_time_in_driver, process.name())
