##
## Traverse the frames associated with each process.
## Calculate the number of state change calls (i.e. non-OpenGL ES draw calls) per OpenGL ES draw call for each frame.
## Report the top 10 most frequently called functions across all standard frames.
##
## This script demonstrates:
## - how to traverse at the frame level (as displayed in the Trace Outline view in MGD)
## - how to use 'raw_input' get ask the user questions
## - how to write that information to a CSV file
##

# You can import the Python standard library into your MGD scripts.
import collections
import os
import sys

# This dictionary will be used to track function frequency as we traverse the process.
function_frequency = collections.defaultdict(int)

# Get our home directory.
home_directory = os.path.expanduser('~')

# We're going to try and create file in our home directory with whatever name the user provides.
filename = raw_input("What would you like to call the report?")

if not filename:
    print("You must supply a filename")
    sys.exit()

# Add the CSV extension if the user didn't.
if not filename.endswith(".csv"):
    filename += ".csv"

frame_report_path = os.path.join(home_directory, filename)

# If the file already exists we don't want to overwrite it.
if not frame_report_path or os.path.exists(frame_report_path):
    print(frame_report_path + " already exists. Exiting to avoid overwriting.")
    sys.exit()

# Open the file for writing.
with open(frame_report_path, 'w') as output_file:

    # Write the headings.
    output_file.write("Process,Frame,Total Number of Calls,Number of Non-OpenGL ES Draw Calls,Number of OpenGL ES Draw Calls\n")

    # Iterate through all the processes in the trace.
    # The "trace" variable is automatically exported in the scripting console when a trace is opened.
    for process in trace.processes():

        monitor.set_task(str(process), len(process.frames()))

        # Iterate through all of the frames in the process.
        for frame in process.frames():

            draw_calls = 0
            non_draw_calls = 0

            # Go through the function calls. Test if they are OpenGL ES draw calls, and if so, log them.
            for function in frame.function_calls():

                if function.is_draw_call():
                    draw_calls += 1

                else:
                    non_draw_calls += 1

                function_frequency[function.name()] += 1

            total = draw_calls + non_draw_calls

            output_file.write("{0},{1},{2},{3},{4}\n".format(process, frame.index(), total, non_draw_calls, draw_calls))

            monitor.increment_progress(1)

    monitor.finish()

    print("Full frame report written to: " + output_file.name)


print("\nMost used function calls:")

# Convert our frequency dictionary into a Counter from the Python standard library.
counter = collections.Counter(function_frequency)

# Counters provide a concise syntax for reporting frequency data.
for fc_name, fc_count in counter.most_common(10):
    print("\t{0} : {1}".format(fc_name, fc_count))
