/*
 * Decompiled with CFR 0.152.
 */
package com.arm.streamline.performanceadvisor.capturedata;

import com.arm.streamline.common.model.counters.CounterDisplay;
import com.arm.streamline.common.model.topology.ProcessingElementReference;
import com.arm.streamline.model.capture.IChartDataProvider;
import com.arm.streamline.model.capture.ISeriesDataProvider;
import com.arm.streamline.performanceadvisor.Application;
import com.arm.streamline.performanceadvisor.capturedata.CaptureData;
import com.arm.streamline.performanceadvisor.common.VectorUtils;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.DoubleSummaryStatistics;
import java.util.List;
import java.util.Optional;
import java.util.logging.Level;

public class CpuClusterProcessor {
    private static final double PERCENTILE = 99.0;
    private static final int WINDOW_SIZE = 50;
    private static final double DIFFERENCE_THRESHOLD = 20.0;
    private static final KnownCpu[] knownCpus = new KnownCpu[]{new KnownCpu("Cortex-A5", 0.8, 1.4), new KnownCpu("Cortex-A7", 1.0, 2.0), new KnownCpu("Cortex-A8", 1.0, 1.5), new KnownCpu("Cortex-A9", 1.0, 2.0), new KnownCpu("Cortex-A15", 1.0, 2.5), new KnownCpu("Cortex-A17", 1.4, 2.5), new KnownCpu("Cortex-A53", 1.0, 2.6), new KnownCpu("Cortex-A55", 1.2, 2.9), new KnownCpu("Cortex-A57", 1.7, 2.2), new KnownCpu("Cortex-A72", 1.7, 2.6), new KnownCpu("Cortex-A73", 1.35, 2.6), new KnownCpu("Cortex-A75", 2.0, 2.4), new KnownCpu("Cortex-A76", 2.0, 3.0), new KnownCpu("Cortex-A77", 2.2, 3.0), new KnownCpu("Exynos-M1", 2.0, 2.6), new KnownCpu("Exynos-M2", 1.7, 2.314), new KnownCpu("Exynos-M3", 2.3, 2.9), new KnownCpu("Exynos-M4", 2.73, 2.73), new KnownCpu("Exynos-M5", 2.73, 2.73)};
    private static final double MAXIMUM_CLOCKSPEED = 3.5E9;
    private static final double MINIMUM_CLOCKSPEED = 5.0E8;
    private double predictedClockspeed;
    private CaptureData captureData;
    private int numCores = 0;
    private Optional<double[]> cycleFrequencies = Optional.empty();
    private Optional<double[]> cyclesAccumulate = Optional.empty();
    private Optional<double[]> criticalUtilisation = Optional.empty();
    private Optional<double[]> activityAvg = Optional.empty();
    private String cpuName;
    private boolean forceCycles;

    public CpuClusterProcessor(CaptureData captureData, IChartDataProvider cyclesGraph, boolean forceCycles) {
        this.captureData = captureData;
        this.forceCycles = forceCycles;
        this.cpuName = this.getCpuNameFromCpuCyclesChartTitle(cyclesGraph.getTitle());
        Application.LOG.fine(() -> "Found cluster with cpu model: " + this.cpuName);
        boolean cyclesPresent = this.populateCyclesData(cyclesGraph);
        boolean useActivity = this.activityAnalysis(this.cpuName, cyclesPresent);
        if (!useActivity) {
            Application.LOG.fine(() -> "Predicted clockspeed after activity analysis: " + this.predictedClockspeed + "Hz");
            this.ensureValidFrequency(this.cpuName);
        }
    }

    private String getCpuNameFromCpuCyclesChartTitle(String cpuCyclesChartTitle) {
        int lengthOfCpuCyclesPrefix = 12;
        if (cpuCyclesChartTitle.length() > lengthOfCpuCyclesPrefix) {
            return cpuCyclesChartTitle.substring(lengthOfCpuCyclesPrefix, cpuCyclesChartTitle.length() - 1);
        }
        return "Unknown";
    }

    private void ensureValidFrequency(String cpuName) {
        double min = 5.0E8;
        double max = 3.5E9;
        KnownCpu[] knownCpuArray = knownCpus;
        int n = knownCpus.length;
        int n2 = 0;
        while (n2 < n) {
            KnownCpu cpu = knownCpuArray[n2];
            if (cpuName.equals(cpu.getName())) {
                Application.LOG.fine(() -> cpuName + " is a known CPU model.");
                min = cpu.getMinClock();
                max = cpu.getMaxClock();
                break;
            }
            ++n2;
        }
        if (this.predictedClockspeed < min) {
            if (Application.LOG.isLoggable(Level.FINE)) {
                Application.LOG.fine("The predicted clockspeed is too low, scaling to valid minimum of " + min + "Hz");
            }
            this.scaleClocks(min / this.predictedClockspeed);
        } else if (this.predictedClockspeed > max) {
            if (Application.LOG.isLoggable(Level.FINE)) {
                Application.LOG.fine("The predicted clockspeed is too high, scaling to valid maximum of " + max + "Hz");
            }
            this.scaleClocks(max / this.predictedClockspeed);
        }
    }

    private boolean populateCyclesData(IChartDataProvider cyclesGraph) {
        Optional<ISeriesDataProvider> series = cyclesGraph.getSeries().stream().filter(s -> s.getName().equals("Cycles")).findFirst();
        if (!series.isPresent()) {
            Application.LOG.fine("Cycles data not found for this cluster");
            return false;
        }
        ArrayList<double[]> cyclesPerCore = new ArrayList<double[]>();
        IChartDataProvider.IChartCoreInformationProvider coreInfo = cyclesGraph.getCoreInformationProvider();
        this.numCores = coreInfo.getChannelCount();
        if (this.numCores < 1) {
            Application.LOG.fine("Cycles data is unreadable for this cluster.");
            return false;
        }
        int binCount = this.captureData.getBinCount();
        this.predictedClockspeed = this.calculatePredictedClockspeed(binCount, series.get(), coreInfo, cyclesPerCore);
        double[] aggregate = new double[binCount];
        double[] critical = new double[binCount];
        int bin = 0;
        while (bin < binCount) {
            int reliableCoreCount = 0;
            int core = 0;
            while (core < this.numCores) {
                double clock = ((double[])cyclesPerCore.get(core))[bin];
                if (clock != Double.NaN && clock <= this.predictedClockspeed) {
                    int n = bin;
                    aggregate[n] = aggregate[n] + clock;
                    ++reliableCoreCount;
                    if (critical[bin] < clock) {
                        critical[bin] = clock;
                    }
                }
                ++core;
            }
            if (reliableCoreCount > 0) {
                int n = bin;
                aggregate[n] = aggregate[n] / (double)reliableCoreCount;
            } else {
                aggregate[bin] = 0.0;
            }
            ++bin;
        }
        critical = VectorUtils.divZeroNan(critical, this.predictedClockspeed);
        aggregate = VectorUtils.mul(aggregate, this.numCores);
        this.cycleFrequencies = Optional.of(aggregate);
        aggregate = VectorUtils.div(aggregate, this.captureData.getSamplesPerSecond());
        this.cyclesAccumulate = Optional.of(aggregate);
        this.criticalUtilisation = Optional.of(critical);
        return true;
    }

    private double calculatePredictedClockspeed(int binCount, ISeriesDataProvider cyclesSeries, IChartDataProvider.IChartCoreInformationProvider coreInfo, List<double[]> cyclesPerCore) {
        ProcessingElementReference[] channels = coreInfo.getChannelDescriptors();
        int anomalyCount = 0;
        double max = 0.0;
        double[] clockSpeeds = new double[this.numCores * binCount];
        int core = 0;
        while (core < this.numCores) {
            double[] coreUtil = (double[])cyclesSeries.getData(channels[core], 0, this.captureData.getBinCount() - 1, null).clone();
            int i = 0;
            while (i < coreUtil.length) {
                int n = i;
                coreUtil[n] = coreUtil[n] * 1000.0;
                if (coreUtil[i] > 3.5E9) {
                    ++anomalyCount;
                    coreUtil[i] = Double.NaN;
                }
                clockSpeeds[i + core * binCount] = coreUtil[i];
                if (coreUtil[i] > max) {
                    max = coreUtil[i];
                }
                ++i;
            }
            cyclesPerCore.add(coreUtil);
            ++core;
        }
        double reliabilityScore = (1.0 - (double)anomalyCount / ((double)this.numCores * (double)binCount)) * 100.0;
        Application.LOG.finer(() -> String.format("CPU clockspeed reliability score %f", reliabilityScore) + "%");
        this.predictedClockspeed = VectorUtils.percentile(clockSpeeds, 99.0 - (double)anomalyCount / (double)binCount * 100.0);
        Application.LOG.fine(() -> String.format("Maximum filtered clockspeed in capture: %fHz", this.predictedClockspeed));
        return this.predictedClockspeed;
    }

    private boolean activityAnalysis(String cpuName, boolean cyclesPresent) {
        boolean useActivity;
        Optional<IChartDataProvider> activityGraph = this.captureData.getChartList().stream().filter(c -> c.getTitle().contains("CPU Activity") && c.getTitle().contains(cpuName)).findFirst();
        if (!activityGraph.isPresent()) {
            return false;
        }
        double maxActivity = 0.0;
        double[] activityAggregate = new double[this.captureData.getBinCount()];
        IChartDataProvider chart = activityGraph.get();
        Optional<ISeriesDataProvider> userSeries = chart.getSeries().stream().filter(c -> c.getName().contains("User")).findFirst();
        if (!userSeries.isPresent()) {
            return false;
        }
        Application.LOG.fine("Enhancing cycles data with relevant activity data");
        ISeriesDataProvider series = userSeries.get();
        IChartDataProvider.IChartCoreInformationProvider coreInfo = chart.getCoreInformationProvider();
        series.getConfig().setDisplay(CounterDisplay.AVERAGE);
        if (coreInfo.getChannelCount() != this.numCores && cyclesPresent) {
            return false;
        }
        ArrayList<double[]> activityPerCore = new ArrayList<double[]>();
        double[] aggregateActivity = new double[this.captureData.getBinCount()];
        ProcessingElementReference[] channels = coreInfo.getChannelDescriptors();
        int core = 0;
        while (core < this.numCores) {
            double[] coreUtil = (double[])series.getData(channels[core], 0, this.captureData.getBinCount() - 1, null).clone();
            activityAggregate = VectorUtils.add(activityAggregate, coreUtil);
            double[] averagedUtil = VectorUtils.slidingWindow(coreUtil, 50);
            activityPerCore.add(averagedUtil);
            aggregateActivity = VectorUtils.add(aggregateActivity, averagedUtil);
            double max = VectorUtils.percentile(averagedUtil, 99.0);
            if (max > maxActivity) {
                maxActivity = max;
            }
            ++core;
        }
        activityAggregate = VectorUtils.divZeroNan(activityAggregate, this.numCores);
        this.activityAvg = Optional.of(activityAggregate);
        if (maxActivity == 0.0) {
            return false;
        }
        boolean bl = useActivity = !cyclesPresent;
        if (cyclesPresent) {
            this.scaleClocks(1.0 / maxActivity);
            boolean bl2 = useActivity = !this.correlationCheck(activityAggregate);
        }
        if (useActivity) {
            Application.LOG.fine("Using activity data for this cluster over cycles data");
            double[] newCriticalUtilisation = new double[this.captureData.getBinCount()];
            for (double[] coreActivity : activityPerCore) {
                newCriticalUtilisation = VectorUtils.max(newCriticalUtilisation, coreActivity);
            }
            this.cyclesAccumulate = Optional.empty();
            this.cycleFrequencies = Optional.empty();
            this.criticalUtilisation = Optional.of(newCriticalUtilisation);
        }
        return useActivity;
    }

    private boolean correlationCheck(double[] avgActivity) {
        if (this.cycleFrequencies.isPresent()) {
            double[] avgCycles = VectorUtils.divZeroNan(this.cycleFrequencies.get(), this.numCores);
            avgCycles = VectorUtils.divZeroNan(avgCycles, this.predictedClockspeed);
            avgActivity = VectorUtils.slidingWindow(avgActivity, 50);
            avgCycles = VectorUtils.slidingWindow(avgCycles, 50);
            double[] correlation = VectorUtils.sub(avgActivity, avgCycles);
            DoubleSummaryStatistics stats = Arrays.stream(correlation).summaryStatistics();
            double avg = stats.getAverage() * 100.0;
            if (!this.forceCycles && (avg > 20.0 || avg < -10.0)) {
                Application.LOG.fine(() -> "Cycles data does not correlate well with activity data, average differance: " + avg);
                return false;
            }
            return true;
        }
        return false;
    }

    private void scaleClocks(double scaleFactor) {
        this.predictedClockspeed *= scaleFactor;
        if (Double.isNaN(this.predictedClockspeed)) {
            this.predictedClockspeed = 0.0;
        }
        if (this.criticalUtilisation.isPresent()) {
            double[] scaled = VectorUtils.divZeroNan(this.criticalUtilisation.get(), scaleFactor);
            this.criticalUtilisation = Optional.of(scaled);
        }
    }

    public Optional<double[]> getCycleFrequencies() {
        return this.cycleFrequencies;
    }

    public Optional<double[]> getCyclesAccumulate() {
        return this.cyclesAccumulate;
    }

    public Optional<double[]> getCriticalUtilisation() {
        return this.criticalUtilisation;
    }

    public Optional<double[]> getAvgActivity() {
        return this.activityAvg;
    }

    public int getCoreCount() {
        return this.numCores;
    }

    public double getPredictedClockspeed() {
        return this.predictedClockspeed;
    }

    public String getName() {
        return this.cpuName;
    }

    static class KnownCpu {
        private String name;
        private double minClock;
        private double maxClock;
        private static final double GHZ_TO_HZ = 1.0E9;

        KnownCpu(String name, double minClock, double maxClock) {
            this.name = name;
            this.minClock = minClock;
            this.maxClock = maxClock;
        }

        String getName() {
            return this.name;
        }

        double getMinClock() {
            return this.minClock * 1.0E9;
        }

        double getMaxClock() {
            return this.maxClock * 1.0E9;
        }
    }
}

